################################################################################
# Performs two one-sided t tests for equivalence of regression coefficients
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9 
# Date: Feb 06, 2026

require("lm.beta")

equivalence.types <- c("delta", "epsilon")

tost.regress <- function(
  formula,
  data=NULL,
  eqv.type=equivalence.types,
  eqv.level=1,
  upper=NA,
  conf.level=0.95,
  relevance=TRUE
  ) {
  if (anyNA(upper)) {
    upper <- abs(eqv.level)
    }
  # Set up formatting for confidence interval column heading
  cl_width <- nchar(toString(100*conf.level))
  if (cl_width == 1) {
    conf.level.display <- paste0("    [",toString(100*conf.level),"% Conf. Interval]")
    }
  if (cl_width == 2) {
    conf.level.display <- paste0("   [",toString(100*conf.level),"% Conf. Interval]")
    }
  if (cl_width == 4) {
    conf.level.display <- paste0(" [",toString(100*conf.level),"% Conf. Interval]")
    }
  if (cl_width == 5) {
    conf.level.display <- paste0("[",toString(100*conf.level),"% Conf. Interval]")
    }
  if (cl_width > 5) {
    conf.level.display <- paste0("   [",toString(100*conf.level),"% CI]")
    }
  # Calculate alpha
  alpha <- (1 - conf.level)
  alpha_display <- round(alpha,cl_width)
  regression_model <- lm(formula,data=data)
  summary_regression_model <- summary(regression_model)
  variables <- attr(terms(formula),"variables")
  n_intercept  <- attr(regression_model$terms, "intercept")
  if (n_intercept == 1) {
    Beta <- lm.beta(regression_model)$standardized.coefficients[-1]
    }
  else {
    Beta <- lm.beta(regression_model)$standardized.coefficients
    }
  names(Beta) <- gsub(".*\\$","",names(Beta))
  n_predictors <- length(regression_model$coefficients) - n_intercept # Modify if >1 left hand side variable
  k            <- n_predictors + n_intercept
  dependent_variable <- pad.left(name.prep(names(regression_model$model)[1]),12)
  dependent_variable_display <- pad.left(gsub(".*\\$","",dependent_variable),12)
  n <- sum(summary_regression_model$df[1:2])
  n_display <- sprintf("%12.0f",n)
  MSE_tot   <- var(regression_model$model[,1])
  df_tot    <- n - n_intercept
  SS_tot    <- MSE_tot*df_tot
  SS_tot_display <- sprintf("%10.9g",SS_tot)
  df_tot_display <- sprintf("%9.0f",df_tot)
  MSE_tot_display <- sprintf("%10.9g",MSE_tot)
  SS_res    <- sum(regression_model$residuals^2)
  df_res    <- regression_model$df.residual
  MSE_res   <- SS_res/df_res
  SS_res_display <- sprintf("%10.9g",SS_res)
  df_res_display <- sprintf("%9.0f",df_res)
  MSE_res_display <- sprintf("%10.9g",MSE_res)
  SS_model  <- SS_tot - SS_res
  df_model  <- n_predictors
  MSE_model <- SS_model/df_model
  SS_model_display <- sprintf("%10.9g",SS_model)
  df_model_display <- sprintf("%9.0f",df_model)
  MSE_model_display <- sprintf("%10.9g",MSE_model)
  Fstat <- 2*(MSE_model/(n_predictors))/MSE_res
  Fstat_display <- sprintf("%11.2f",Fstat)
  numerator_df <- n_predictors
  denominator_df <- n - k
  FDist_display <- pad.right( paste0("F(",trimws(toString(numerator_df)),",",trimws(toString(denominator_df)),")"), 16)
  FProb_display <- pad.left((sprintf("%6.4f",pf(Fstat, numerator_df, denominator_df, lower.tail=FALSE))),11, strip0=FALSE)
  Rsq       <- 1 - (SS_res/SS_tot)
  Rsq_display <- pad.left(sprintf("%6.4f",Rsq),11, strip0=FALSE)
  Rsq_adj   <- 1 - (1 - Rsq)*(n - n_intercept)/(n - k)
  Rsq_adj_display <- pad.left(sprintf("%6.4f",Rsq_adj),11, strip0=FALSE)
  RMSE      <- sqrt(MSE_res)
  RMSE_display <- sprintf("%11.4f",RMSE)
  # Create output bars and table headers
  top.bar <- function(x) {
    return(paste0(
      paste0(rep("\U2500",13),collapse=""),
      "\U252C",
      paste0(rep("\U2500",x),collapse=""), collapse=""))}
  mid.bar <- function(x) {
   return(paste0(
      paste0(rep("\U2500",13),collapse=""),
      "\U253C",
      paste0(rep("\U2500",x),collapse=""), collapse=""))}
  mid.bar1 <- paste0(
      paste0(rep("\U2500",13),collapse=""),
      "\U253C",
      paste0(rep("\U2500",34),collapse=""),
      pad.spaces(3), collapse="")
  bottom.bar <- function(x) {
   return(paste0(
      paste0(rep("\U2500",13),collapse=""),
      "\U2534",
      paste0(rep("\U2500",x),collapse=""),
      collapse=""))}
  table.title1 <- paste0(
      pad.spaces(7),
      "Model \U2502",
      pad.spaces(7),
      "SS",
      pad.spaces(11),
      "df",
      pad.spaces(7),
      "MS",
      pad.spaces(6),
      "Number of obs",
      pad.spaces(3),
      "=",
      n_display, 
      sep="", collapse="")
  table.title2 <- paste0(
      dependent_variable_display,
      " \U2502 Coefficient",
      pad.spaces(2),
      "Std. err.",
      pad.spaces(5),
      "t",
      pad.spaces(3),
      "P(|T|>|t|) ",
      conf.level.display, sep="", collapse="")
  table.title3a <- paste0(
      dependent_variable_display,
      " \U2502 Coefficient",
      pad.spaces(2),
      "Std. Err.",
      pad.spaces(4),
      "t1",
      pad.spaces(3),
      "P(T>t1)",
      pad.spaces(4),
      "t2",
      pad.spaces(5),
      "P(T>t2)",
      pad.spaces(3),
      "eqv.type",
      pad.spaces(3),
      "eqv.level",sep="")
  table.title3b <- paste0(
      dependent_variable_display,
      " \U2502 Coefficient",
      pad.spaces(2),
      "Std. Err.",
      pad.spaces(4),
      "t1",
      pad.spaces(4),
      "P(T>t1)",
      pad.spaces(5),
      "t2",
      pad.spaces(4),
      "P(T>t2)",
      pad.spaces(3),
      "eqv.type",
      pad.spaces(2),
      "lower",pad.spaces(4),
      "upper",sep="")
  if (relevance) {
    positivist_decisions <- rep("Not Reject", times=k)
    negativist_decisions <- rep("Not Reject", times=k)
    rlang::inform(message="\nRegression tests for difference")
    rlang::inform(message=top.bar(34))
    rlang::inform(message=table.title1)
    rlang::inform(message=paste0(mid.bar1,FDist_display,"= ",Fstat_display))
    rlang::inform(message=paste0(pad.spaces(7),"Model \U2502  ",SS_model_display," ",df_model_display,"  ",MSE_model_display,pad.spaces(3),"Prob > F",pad.spaces(8),"= ",FProb_display))
    rlang::inform(message=paste0(pad.spaces(4),"Residual \U2502  ",SS_res_display," ",df_res_display,"  ",MSE_res_display,pad.spaces(3),"R-squared",pad.spaces(7),"= ",Rsq_display))
    rlang::inform(message=paste0(mid.bar1,"Adj R-squared   = ",Rsq_adj_display))
    rlang::inform(message=paste0(pad.spaces(7),"Total \U2502  ",SS_tot_display," ",df_tot_display,"  ",MSE_tot_display,pad.spaces(3),"Root MSE",pad.spaces(8),"= ",RMSE_display,"\n"))
    rlang::inform(message=top.bar(66))
    rlang::inform(message=table.title2)
    rlang::inform(message=mid.bar(66))
    table_range <- (1:n_predictors)+n_intercept
    T_pos <- c()
    P_pos <- c()
    for (p in table_range) {
      coefficient <- regression_model$coefficients[p]
      se          <- summary_regression_model$coefficients[p,2]
      t_crit      <- qt((1-conf.level)/2,df=n-k,lower.tail=FALSE)
      ci_lb       <- coefficient - t_crit*se
      ci_ub       <- coefficient + t_crit*se
      ci_lb_display <- pad.left(sprintf("%9.7g",ci_lb),9)
      ci_ub_display <- pad.left(sprintf("%9.7g",ci_ub),9)
#      name_display <- pad.left(name.prep(names(regression_model$coefficients))[p + name_offset],12)
      name_display <- pad.left(name.prep(names(regression_model$coefficients))[p],12)
      coef_display <- pad.left(sprintf("%9.7g",coefficient),10)
      se_display <- pad.left(sprintf("%8.7g",se),8)
      t_pos <- summary_regression_model$coefficients[p,3]
      t_display  <- pad.left(sprintf("%7.2f",t_pos),8)
      p_pos <- summary_regression_model$coefficients[p,4]
      if (p_pos <= alpha) {
        positivist_decisions[p] <- "  Reject  "
        }
      p_display  <- format.extreme.p.vals(p_pos, regress=TRUE)
      rlang::inform(message=paste0(name_display," \U2502 ", coef_display,pad.spaces(3),se_display,pad.spaces(0),t_display,pad.spaces(4),p_display,pad.spaces(4),ci_lb_display,pad.spaces(3),ci_ub_display))
      T_pos <- c(T_pos, t_pos)
      P_pos <- c(P_pos, p_pos)
      }
    if (n_intercept == 1) {
      coefficient   <- regression_model$coefficients[1]
      se            <- summary_regression_model$coefficients[1,2]
      t_crit        <- qt((1-conf.level)/2,df=n-k,lower.tail=FALSE)
      ci_lb         <- coefficient - t_crit*se
      ci_ub         <- coefficient + t_crit*se
      ci_lb_display <- pad.left(sprintf("%9.7g",ci_lb),9)
      ci_ub_display <- pad.left(sprintf("%9.7g",ci_ub),9)
      coef_display  <- pad.left(sprintf("%9.7g",coefficient),10)
      se_display    <- pad.left(sprintf("%8.7g",se),8)
      t_pos         <- summary_regression_model$coefficients[1,3]
      t_display     <- pad.left(sprintf("%7.2f",t_pos),8)
      p_pos         <- summary_regression_model$coefficients[1,4]
      if (p_pos <= alpha) {
        positivist_decisions[1] <- "  Reject  "
        }
      p_display  <- format.extreme.p.vals(p_pos, regress=TRUE)
      rlang::inform(message=paste0("       _cons \U2502 ", coef_display,pad.spaces(3),se_display,pad.spaces(0),t_display,pad.spaces(4),p_display,pad.spaces(4),ci_lb_display,pad.spaces(3),ci_ub_display))
      T_pos <- c(T_pos, t_pos)
      P_pos <- c(P_pos, p_pos)
      }
    rlang::inform(message=paste0(bottom.bar(66),"\n"))
    }
  if (length(eqv.type) == 1) {
    eqv.type <- rep(eqv.type, times=k)
    }
  if (length(eqv.level) == 1) {
    eqv.level <- rep(eqv.level, times=k)
    }
  if (length(upper) == 1) {
    upper <- rep(upper, times=k)
    }
  rlang::inform(message="Regression tests for equivalence")
  if (all(upper == abs(eqv.level))) {
    rlang::inform(message=top.bar(80))
    rlang::inform(message=table.title3a)
    rlang::inform(message=mid.bar(80))
    }
   else {
    rlang::inform(message=top.bar(86))
    rlang::inform(message=table.title3b)
    rlang::inform(message=mid.bar(86))
    }
  table_range <- (1:n_predictors)+n_intercept
  B <- c()
  SE <- c()
  T1 <- c()
  T2 <- c()
  P1 <- c()
  P2 <- c()
  eqv_lower <- c()
  eqv_upper <- c()
  eqv_types <- c()
  mark  <- ""
  markl <- ""
  marku <- ""
  note  <- ""
  notel <- ""
  noteu <- ""
  for (p in table_range) {
    coefficient <- summary_regression_model$coefficients[p,1]
    se          <- summary_regression_model$coefficients[p,2]
    t_pos       <- summary_regression_model$coefficients[p,3]
#    name_display <- pad.left(name.prep(names(regression_model$coefficients))[p + name_offset],12)
    name_display <- pad.left(name.prep(names(regression_model$coefficients))[p],12)
    coef_display <- pad.left(sprintf("%9.7g",coefficient),10)
    se_display <- pad.left(sprintf("%8.7g",se),8)
    eqv_lower <- c(eqv_lower, eqv.level[p])
    if (eqv.type[p]=="delta") {
      eqv_types <- c(eqv_types,"\U0394")
      eqv.type_display <- "   \U0394   "
      if (upper[p] == abs(eqv.level[p])) {
        eqv_upper <- c(eqv_upper, abs(eqv.level[p]))
        t1 <- (eqv.level[p] - coefficient)/se
        }
      if (upper[p] != abs(eqv.level[p])) {
        eqv_upper <- c(eqv_upper, upper[p])
        t1 <- (upper[p] - coefficient)/se
        }
      t2 <- (coefficient + abs(eqv.level[p]))/se
      }
    if (eqv.type[p]=="epsilon") {
      eqv_types <- c(eqv_types,"\U03B5")
      eqv.type_display <- paste0("   \u03B5   ")
      if (upper[p] == abs(eqv.level[p])) {
        eqv_upper <- c(eqv_upper, abs(eqv.level[p]))
        t1 <- eqv.level[p] - t_pos
        }
      if (upper[p] != abs(eqv.level[p])) {
        eqv_upper <- c(eqv_upper, upper[p])
        t1 <- upper[p] - t_pos
        }
      t2 <- t_pos + abs(eqv.level[p])
      }
    eqv.level_display <- sprintf("%8.3f",eqv.level[p])
    lower_display <- pad.left(sprintf("%8.2f",-1*abs(eqv.level[p])),8)
    upper_display <- pad.left(sprintf("%8.2f",upper[p]),8)
    p1 <- pt(t1, df=(n-k), lower.tail=FALSE)
    p2 <- pt(t2, df=(n-k), lower.tail=FALSE)
    if (relevance) {
      if (p1 <= alpha & p2 <= alpha) {
        negativist_decisions[p] <- "  Reject  "
        }
      }
    t1_display  <- pad.left(sprintf("%7.2f",t1),8)
    p1_display <- format.extreme.p.vals(p1,regress=TRUE)
    t2_display  <- pad.left(sprintf("%7.2f",t2),8)
    p2_display <- format.extreme.p.vals(p2,regress=TRUE)
    if (all(upper == abs(eqv.level))) {
      # Produce mark and note for an impossible rejection threshold
      if (eqv.type[p]=="delta") {
        impossible_warning <- impossibledelta(eqv.level[p], eqv.level[p], se, n-k, 1-conf.level)
        }
      if (eqv.type[p]=="epsilon") {
        impossible_warning <- impossibleepsilon(eqv.level[p], eqv.level[p], n-k, 1-conf.level)
        }
      mark <- c(mark, impossible_warning[1])
      note <- c(note, impossible_warning[2])
      rlang::inform(message=paste0(
          name_display,
          " \U2502 ", 
          coef_display,
          pad.spaces(3),
          se_display,
          pad.spaces(0), 
          t1_display,
          pad.spaces(2),
          p1_display,
          pad.spaces(0),
          t2_display,
          pad.spaces(3),
          p2_display,
          pad.spaces(3),
          eqv.type_display,
          pad.spaces(3), 
          eqv.level_display, 
          tail(mark, n=1)))
      }
     else {
      # Produce mark and note for an impossible rejection threshold
      if (eqv.type[p]=="delta") {
        impossible_warning <- impossibledelta(eqv.level[p], upper[p], se, n-k, 1-conf.level)
        }
      if (eqv.type[p]=="epsilon") {
        impossible_warning <- impossibleepsilon(eqv.level[p], upper[p], n-k, 1-conf.level)
        }
      markl <- c(markl, impossible_warning[3])
      notel <- c(notel, impossible_warning[4])
      marku <- c(marku, impossible_warning[5])
      noteu <- c(noteu, impossible_warning[6])
      mark_pad <- 1
      if (tail(markl, n=1) != "") {
        mark_pad <- 0
        }
      rlang::inform(message=paste0(
          name_display,
          " \U2502 ", 
          coef_display,
          pad.spaces(3),
          se_display,
          pad.spaces(0), 
          t1_display,
          pad.spaces(3),
          p1_display,
          pad.spaces(0),
          t2_display,
          pad.spaces(3),
          p2_display,
          pad.spaces(3),
          eqv.type_display,
          pad.spaces(0),
          lower_display,
          tail(markl, n=1),
          pad.spaces(mark_pad),
          upper_display,
          tail(marku, n=1)))
      }
    B <- c(B, coefficient)
    SE <- c(SE, se)
    T1 <- c(T1, t1)
    T2 <- c(T2, t2)
    P1 <- c(P1, p1)
    P2 <- c(P2, p2)
    }
  if (n_intercept == 1) {
    coefficient <- summary_regression_model$coefficients[1,1]
    se          <- summary_regression_model$coefficients[1,2]
    t_pos       <- summary_regression_model$coefficients[1,3]
    coef_display <- pad.left(sprintf("%9.7g",coefficient),10)
    se_display <- pad.left(sprintf("%8.7g",se),8)
    eqv_lower <- c(eqv_lower, eqv.level[1])
    if (eqv.type[1]=="delta") {
      eqv_types <- c(eqv_types,"\U0394")
      eqv.type_display <- "   \U0394   "
      if (upper[1] == abs(eqv.level[1])) {
        eqv_upper <- c(eqv_upper, abs(eqv.level[1]))
        t1 <- (eqv.level[1] - coefficient)/se
        }
      if (upper[1] != abs(eqv.level[1])) {
        eqv_upper <- c(eqv_upper, upper[1])
        t1 <- (upper[1] - coefficient)/se
        }
      t2 <- (coefficient + abs(eqv.level[1]))/se
      }
    if (eqv.type[1]=="epsilon") {
      eqv_types <- c(eqv_types,"\U03B5")
      eqv.type_display <- paste0("   \u03B5   ")
      if (upper[1] == abs(eqv.level[1])) {
        eqv_upper <- c(eqv_upper, abs(eqv.level[1]))
        t1 <- eqv.level[1] - t_pos
        }
      if (upper[1] != abs(eqv.level[1])) {
        eqv_upper <- c(eqv_upper, upper[1])
        t1 <- upper[1] - t_pos
        }
      t2 <- t_pos + abs(eqv.level[1])
      }
    eqv.level_display <- sprintf("%8.3f",eqv.level[1])
    lower_display <- pad.left(sprintf("%8.2f",-1*abs(eqv.level[1])),8)
    upper_display <- pad.left(sprintf("%8.2f",upper[1]),8)
    p1 <- pt(t1, df=(n-k), lower.tail=FALSE)
    p2 <- pt(t2, df=(n-k), lower.tail=FALSE)
    if (relevance) {
      if (p1 <= alpha & p2 <= alpha) {
        negativist_decisions[1] <- "  Reject  "
        }
      }
    t1_display  <- pad.left(sprintf("%7.2f",t1),8)
    p1_display <- format.extreme.p.vals(p1,regress=TRUE)
    t2_display  <- pad.left(sprintf("%7.2f",t2),8)
    p2_display <- format.extreme.p.vals(p2,regress=TRUE)
    if (all(upper == abs(eqv.level))) {
      # Produce mark and note for an impossible rejection threshold
      if (eqv.type[1]=="delta") {
        impossible_warning <- impossibledelta(eqv.level[1], eqv.level[1], se, n-k, 1-conf.level)
        }
      if (eqv.type[1]=="epsilon") {
        impossible_warning <- impossibleepsilon(eqv.level[1], eqv.level[1], n-k, 1-conf.level)         }
      mark <- c(mark, impossible_warning[1])
      note <- c(note, impossible_warning[2])
      rlang::inform(message=paste0(
          pad.spaces(7),
          "_cons \U2502 ", 
          coef_display,
          pad.spaces(3),
          se_display,
          pad.spaces(0),
          t1_display,
          pad.spaces(2),
          p1_display,
          pad.spaces(0),
          t2_display,
          pad.spaces(3),
          p2_display,
          pad.spaces(3),
          eqv.type_display,
          pad.spaces(3),
          eqv.level_display,
          tail(mark, n=1)))
      }
     else {
      # Produce mark and note for an impossible rejection threshold
      if (eqv.type[1]=="delta") {
        impossible_warning <- impossibledelta(eqv.level[1], upper[1], se, n-k, 1-conf.level)
        }
      if (eqv.type[1]=="epsilon") {
        impossible_warning <- impossibleepsilon(eqv.level[1], upper[1], n-k, 1-conf.level)
        }
      markl <- c(markl, impossible_warning[3])
      notel <- c(notel, impossible_warning[4])
      marku <- c(marku, impossible_warning[5])
      noteu <- c(noteu, impossible_warning[6])
      mark_pad <- 1
      if (tail(markl, n=1) != "") {
        mark_pad <- 0
        }
      rlang::inform(message=paste0(
          pad.spaces(7),
          "_cons \U2502 ",
          coef_display,
          pad.spaces(3),
          se_display,
          pad.spaces(0),
          t1_display,
          pad.spaces(3),
          p1_display,
          pad.spaces(0),
          t2_display,
          pad.spaces(3),
          p2_display,
          pad.spaces(3),
          eqv.type_display,
          pad.spaces(0),
          lower_display,
          tail(markl, n=1),
          pad.spaces(mark_pad),
          upper_display),
          tail(marku, n=1))
      }
    B <- c(B, coefficient)
    SE <- c(SE, se)
    T1 <- c(T1, t1)
    T2 <- c(T2, t2)
    P1 <- c(P1, p1)
    P2 <- c(P2, p2)
    }
  if (all(upper == abs(eqv.level))) {
    rlang::inform(message=bottom.bar(80))
    # If any warning mark was reported, include the first such warning note
    if ("\U1D43" %in% mark){
      rlang::inform(message=note[which(note != "")[1]])
      }
    }
   else {
    rlang::inform(message=bottom.bar(86))
    # If any warning markl was reported, include the first such warning notel
    if ("\U1D38" %in% markl){
      rlang::inform(message=notel[which(notel != "")[1]])
      }
    # If any warning marku was reported, include the first such warning noteu
    if ("\U1D41" %in% marku){
      rlang::inform(message=noteu[which(noteu != "")[1]])
      }
    }
  names(eqv_lower) <- eqv_types
  names(eqv_upper) <- eqv_types
  if (relevance) {
    relevance_conclusion <- rep("", times=k)
    rlang::inform(message=paste0("\nRegression relevance tests (\U03B1 = ",alpha_display,")"))
    if (all(upper == abs(eqv.level))) {
      rlang::inform(message=top.bar(67))
      rlang::inform(message=paste0(dependent_variable_display," \U2502 eqv.type",pad.spaces(3),"eqv.level",pad.spaces(6),"Ho+",pad.spaces(9),"Ho-",pad.spaces(11),"Conclude"))
      rlang::inform(message=mid.bar(67))
      }
     else {
      rlang::inform(message=top.bar(73))
      rlang::inform(message=paste0(dependent_variable_display," \U2502 eqv.type",pad.spaces(3),"lower",pad.spaces(4),"upper",pad.spaces(7),"Ho+",pad.spaces(9),"Ho-",pad.spaces(11),"Conclude"))
      rlang::inform(message=mid.bar(73))
      }
    for (p in table_range) {
      if (positivist_decisions[p] == "  Reject  " & negativist_decisions[p] == "  Reject  ") {
      relevance_conclusion[p] <- "Trivial Difference "
      }
      if (positivist_decisions[p] == "  Reject  " & negativist_decisions[p] == "Not Reject") {
        relevance_conclusion[p] <- "Relevant Difference"
        }
      if (positivist_decisions[p] == "Not Reject" & negativist_decisions[p] == "  Reject  ") {
        relevance_conclusion[p] <- "    Equivalence    "
        }
      if (positivist_decisions[p] == "Not Reject" & negativist_decisions[p] == "Not Reject") {
        relevance_conclusion[p] <- "   Inconclusive  "
        }
      if (eqv.type[p]=="delta") {
        eqv.type_display <- "   \U0394   "
        }
      if (eqv.type[p]=="epsilon") {
        eqv.type_display <- paste0("   \u03B5   ")
        }
#      name_display <- pad.left(name.prep(names(regression_model$coefficients))[p + name_offset],12)
      name_display <- pad.left(name.prep(names(regression_model$coefficients))[p],12)
      eqv.level_display <- sprintf("%8.2f",eqv.level[p])
      lower_display <- pad.left(sprintf("%8.2f",-1*abs(eqv.level[p])),8)
      upper_display <- pad.left(sprintf("%8.2f",upper[p]),8)
      if (all(upper == abs(eqv.level))) {
        rlang::inform(message=paste0(name_display," \U2502  ", eqv.type_display, pad.spaces(1), eqv.level_display, pad.spaces(5), positivist_decisions[p], pad.spaces(2), negativist_decisions[p],pad.spaces(3),relevance_conclusion[p]))
        }
       else {
        rlang::inform(message=paste0(name_display," \U2502  ", eqv.type_display, lower_display, pad.spaces(1), upper_display, pad.spaces(3), positivist_decisions[p], pad.spaces(2), negativist_decisions[p],pad.spaces(3),relevance_conclusion[p]))
        }
      }
    if (n_intercept == 1) {
      if (positivist_decisions[1] == "  Reject  " & negativist_decisions[1] == "  Reject  ") {
        relevance_conclusion[1] <- "Trivial Difference "
        }
      if (positivist_decisions[1] == "  Reject  " & negativist_decisions[1] == "Not Reject") {
        relevance_conclusion[1] <- "Relevant Difference"
        }
      if (positivist_decisions[1] == "Not Reject" & negativist_decisions[1] == "  Reject  ") {
        relevance_conclusion[1] <- "    Equivalence    "
        }
      if (positivist_decisions[1] == "Not Reject" & negativist_decisions[1] == "Not Reject") {
        relevance_conclusion[1] <- "   Inconclusive    "
        }
      if (eqv.type[1]=="delta") {
        eqv.type_display <- "   \U0394   "
        }
      if (eqv.type[1]=="epsilon") {
        eqv.type_display <- paste0("   \u03B5   ")
        }
      eqv.level_display <- sprintf("%8.2f",eqv.level[1])
      lower_display <- pad.left(sprintf("%8.2f",-1*abs(eqv.level[1])),8)
      upper_display <- pad.left(sprintf("%8.2f",upper[1]),8)
      if (all(upper == abs(eqv.level))) {
        rlang::inform(message=paste0("       _cons \U2502  ", eqv.type_display, pad.spaces(1), eqv.level_display, pad.spaces(5), positivist_decisions[1], pad.spaces(2), negativist_decisions[1],pad.spaces(3),relevance_conclusion[1]))
        }
       else {
        rlang::inform(message=paste0("       _cons \U2502  ", eqv.type_display, lower_display, pad.spaces(1), upper_display, pad.spaces(3), positivist_decisions[1], pad.spaces(2), negativist_decisions[1],pad.spaces(3),relevance_conclusion[1]))
        }
      }
    if (all(upper == abs(eqv.level))) {
      rlang::inform(message=bottom.bar(67))
     }
     else {
      rlang::inform(message=bottom.bar(73))
     }
    }
###############################################################################
# Program end. Close up shop and return things.                               #
###############################################################################
  out <- list() 
  # Prepare return stuff
  out$N           <- n
  out$df_m        <- n_predictors
  out$df_r        <- n - k
  out$F           <- Fstat
  out$r2          <- Rsq
  out$rmse        <- RMSE
  out$mss         <- SS_model
  out$rss         <- SS_res
  out$r2_a        <- Rsq_adj
  out$alpha       <- 1 - conf.level
  out$T1          <- T1
  out$T2          <- T2
  if (relevance) {
    out$T_pos     <- T_pos
    }
  out$P1          <- P1
  out$P2          <- P2
  if (relevance) {
    out$P_pos     <- P_pos
    }
  out$B           <- B
  out$SE          <- SE
  if (n_intercept==0) {
    out$V         <- vcov(regression_model)
    }
  if (n_intercept==1) {
    out$V         <- stata_vcm(regression_model)
    }
  out$Beta        <- Beta
  out$thresholds_lower   <- eqv_lower
  out$thresholds_upper   <- eqv_upper
  if (relevance) {
    out$conclusions <- trimws(relevance_conclusion)
    }
  invisible(out)
  }
