% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lm.mle.R
\name{lm.mle}
\alias{lm.mle}
\title{Linear Regression via MLE}
\usage{
lm.mle(
  formula,
  data,
  subset,
  ln.var.v = NULL,
  technique = c("bfgs"),
  lmtol = 1e-05,
  reltol = 1e-12,
  maxit = 199,
  optim.report = 1,
  optim.trace = 10,
  print.level = 0,
  digits = 4,
  only.data = FALSE,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{formula} specifying the regression:
typically \code{y ~ x1 + ...}, where \code{y} is the dependent variable and
the \code{x}'s are regressors.}

\item{data}{an optional \code{data.frame} containing the variables referenced in
\code{formula}. If not found in \code{data}, variables are taken from
\code{environment(formula)}.}

\item{subset}{an optional logical or numeric vector specifying the subset of observations
to be used in estimation.}

\item{ln.var.v}{optional one-sided formula; e.g. \code{ln.var.v ~ z1 + z2}. When provided,
the error variance is modeled as \eqn{\log(\sigma_i^2) = w_i^\top \gamma_v}.
If \code{NULL}, the variance is homoskedastic.}

\item{technique}{character vector specifying the preferred optimization routine(s) in order of
preference. Recognized keywords (for future implementation) include \code{"bfgs"}
\code{"bhhh"}, \code{"nm"} (Nelder–Mead), \code{"bfgs"},
and \code{"cg"}. Default is \code{"bfgs"}. This scaffold records but does not
execute the chosen routine.}

\item{lmtol}{numeric. Convergence tolerance based on scaled gradient (when applicable).
Default \code{1e-5}.}

\item{reltol}{numeric. Relative convergence tolerance for likelihood maximization.
Default \code{1e-12}.}

\item{maxit}{integer. Maximum number of iterations for the optimizer. Default \code{199}.}

\item{optim.report}{integer. Verbosity level for reporting progress (if implemented). Default \code{1}.}

\item{optim.trace}{integer. Trace level for optimization (if implemented). Default \code{1}.}

\item{print.level}{integer. Printing level for summaries. Default \code{0}.}

\item{digits}{integer. Number of digits for printing. Default \code{4}.}

\item{only.data}{logical. If \code{TRUE}, returns only constructed data/matrices without
estimation. Default \code{FALSE}.}

\item{...}{additional arguments reserved for future methods (e.g., bounds, penalties).}
}
\value{
A list of class \code{"snreg"} containing:
  \item{\code{par}}{Numeric vector of MLE parameter estimates.}
  \item{\code{value}}{Maximized log-likelihood.}
  \item{\code{ll}}{Maximized log-likelihood (alias).}
  \item{\code{counts}}{Number of function evaluations (from \code{optim}).}
  \item{\code{convergence}}{Convergence code from \code{optim}.}
  \item{\code{message}}{Message returned by \code{optim}.}
  \item{\code{hessian}}{Observed Hessian matrix at optimum.}
  \item{\code{coef}}{Named coefficient vector; equal to \code{par}.}
  \item{\code{vcov}}{Variance–covariance matrix \code{solve(-hessian)}.}
  \item{\code{sds}}{Standard errors: \code{sqrt(diag(vcov))}.}
  \item{\code{ctab}}{Coefficient table with columns: \code{Estimate}, 
      \code{Std.Err}, \code{Z value}, \code{Pr(>z)}.}
  \item{\code{esample}}{Logical vector: observations used in estimation.}
  \item{\code{n}}{Number of observations in estimation sample.}

The object inherits the default
\code{optim} components and is assigned class \code{"snreg"}.
}
\description{
\code{lm.mle} fits a linear regression model by maximum likelihood, allowing
for optional multiplicative heteroskedasticity in the disturbance variance via
a log-linear specification provided through \code{ln.var.v}.
}
\details{
Linear Model by Maximum Likelihood (with optional heteroskedasticity)


This function fits a maximum-likelihood linear model.

The model is
\deqn{y_i = x_i^\top \beta + \varepsilon_i,\quad \varepsilon_i \sim \mathcal{N}(0, \sigma_i^2).}
When \code{ln.var.v} is supplied, the variance follows
\deqn{\log(\sigma_i^2) = w_i^\top \gamma_v,}
otherwise \eqn{\sigma_i^2 = \sigma^2} is constant (homoskedastic).

This function:
\itemize{
  \item Builds the model frame and \code{X}, \code{y}.
  \item Builds \code{Zv} for the log-variance index when \code{ln.var.v} is provided.
  \item Returns a structured object with placeholders for \code{coef}, \code{vcov}, \code{loglik}.
}
Insert your MLE engine to estimate \eqn{\beta}, and (optionally) \eqn{\sigma^2} or
\eqn{\gamma_v}; compute standard errors via AIM/OPG as required by \code{vcetype}.
}
\examples{

library(snreg)

data("banks07")
head(banks07)

# Translog cost function specification

spe.tl <- log(TC) ~ (log(Y1) + log(Y2) + log(W1) + log(W2))^2 +
  I(0.5 * log(Y1)^2) + I(0.5 * log(Y2)^2) +
  I(0.5 * log(W1)^2) + I(0.5 * log(W2)^2)

# Specification 1: homoskedastic noise (ln.var.v = NULL)

formSV <- NULL   # variance equation; constant variance

m1 <- lm.mle(
  formula   = spe.tl,
  data      = banks07,
  ln.var.v  = formSV
)

summary(m1)

# Specification 2: heteroskedastic

formSV <- ~ log(TA)      # variance equation; heteroskedastic noise (variance depends on TA)

m2 <- lm.mle(
  formula   = spe.tl,
  data      = banks07,
  ln.var.v  = formSV
)

summary(m2)

}
\seealso{
\code{\link{snsf}}, \code{\link{snreg}}
}
\keyword{heteroskedasticity}
\keyword{maximum-likelihood}
\keyword{regression}
