% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/speci.R
\name{speci.VAR}
\alias{speci.VAR}
\title{Criteria on the lag-order and break period(s)}
\usage{
speci.VAR(
  x,
  lag_set = 1:10,
  dim_m = FALSE,
  trim = 0.15,
  type_break = "const",
  add_dummy = FALSE,
  n.cores = 1
)
}
\arguments{
\item{x}{VAR object of class '\code{varx}' or any other 
that will be \link[=as.varx]{coerced} to '\code{varx}'. Specifically for 
\strong{vars}' \code{\link[vars]{VAR}}, use \code{p = min(lag_set)} 
or simply \code{p=1} such that the customized \code{$D} from the coerced 
'\code{varx}' object contains no \code{NA} in the effective sample.}

\item{lag_set}{Vector. Set of candidates for the lag-order \eqn{p}. 
If only a single integer is provided, the criteria just reflect 
the variation of det\eqn{(\hat{U}_{\tau} \hat{U}_{\tau}')} uniformly and 
determine the break period(s) \eqn{\tau} unanimously as \eqn{\hat{\tau} = }
arg min det\eqn{(\hat{U}_{\tau} \hat{U}_{\tau}')} under the given \eqn{p}.}

\item{dim_m}{Integer. Number of breaks in the deterministic term to consider.
If \code{FALSE} (the default), the criteria determine only 
the lag-order \eqn{p} just like \strong{vars}' \code{\link[vars]{VARselect}}.}

\item{trim}{Numeric. Either a numeric value \eqn{h \in (p_{max}/T, 1/m)} that 
defines the minimal fraction relative to the total sample size \eqn{T} or 
an integer that defines the minimal number of observations in each sub-sample. 
For example, \eqn{h=0.15} (the default) specifies the window 
\eqn{[0.15 \cdot T, 0.85 \cdot T]} that is often used 
as the set of candidates for \eqn{m=1} single period \eqn{\tau_1}.}

\item{type_break}{Character. Whether the \eqn{m} common breaks pertain the 
'\code{const}' (the default), the linear '\code{trend}', or '\code{both}'. 
Adds the period-specific \link[=as.t_D]{deterministic term} activated 
during \eqn{\tau}.}

\item{add_dummy}{Logical. If \code{TRUE} (not the default), accompanying 
impulse dummies activated in \eqn{\tau + (0, \ldots, p-1)} are added to each break.}

\item{n.cores}{Integer. Number of allocated processor cores.
Note that parallel processing is exclusively activated for the combined 
determination of lag-order \eqn{p} and break period(s) \eqn{\tau} only.}
}
\value{
A list of class '\code{speci}', which contains the elements: 
\item{df}{A '\code{data.frame}' of \eqn{(1+m) + 4} columns for all admissible 
  combinations of candidate \eqn{(p, \tau)} and their values of 
  \eqn{AIC(p, \tau)}, \eqn{HQC(p, \tau)}, \eqn{SIC(p, \tau)}, and \eqn{FPE(p, \tau)}.}
\item{selection}{A \eqn{(1+m) \times 4} matrix of the specification pairs 
  \eqn{(p^*, \tau^*)} suggested by the global minimum of the AIC (Akaike 1969), 
  HQC (Hannan, Quinn 1979), SIC (Schwarz 1978), and FPE respectively.}
\item{args_speci}{List of characters and integers indicating the specifications that have been used.}
}
\description{
Determines the lag-order \eqn{p} and break period(s) \eqn{\tau} 
  jointly via information criteria on the OLS-estimated VAR model for a given 
  number of breaks. These \eqn{m} breaks are common to all \eqn{K} equations 
  of the system and partial, as pertaining the 
  \link[=as.t_D]{deterministic term} only.
}
\details{
The literature on structural breaks in time series deals mostly with 
  the determination of the number \eqn{m} and position \eqn{\tau} of breaks 
  (e.g. Bai, Perron 1998 and 2003), but leaves the lag-order \eqn{p} aside. 
  For example, under a given \eqn{p}, Luetkepohl et al. (2004) use a full-rank 
  VAR in levels to determine \eqn{m=1} common break period \eqn{\tau_1} 
  and subsequently perform cointegration analysis with \code{\link{coint.SL}} 
  (which actually provides \eqn{p}-values for up to \eqn{m=2}). 
  Note yet that the lag-order of a VECM is usually determined via 
  information criteria of a full-rank VAR in levels alike.
  
  \code{\link{speci.VAR}} combines Bai, Perron (2003) and Approach 3 of Yang (2002)
  into a global minimization of information criteria on the pair \eqn{(p,\tau)}. 
  Specifically, Yang (2002:378, Ch.2.2) estimates all candidate VAR models by 
  OLS and then determines their optimal lag-order \eqn{p^*} and \eqn{m=1} break 
  period \eqn{\tau^*} jointly via the global minimum of the information criteria. 
  Bai and Perron (2003, Ch.3) determine 
  \eqn{\tau^* = (\tau_1^*, \ldots, \tau_m^*)} of multiple breaks via the 
  minimum sum of squared residuals from a single-equation model \eqn{(K=1)}. 
  They use dynamic programming to reduce the number of least-squares operations. 
  Although adapting their streamlined set of admissible combinations for \eqn{\tau}, 
  \code{\link{speci.VAR}} yet resorts to (parallelized brute-force) OLS estimation 
  of all candidate VAR models and therewith circumvents issues of correct 
  initialization and iterative updating for the models with partial breaks.
}
\examples{
### extend basic example in "urca" ###
library("urca")
library("vars")
data("denmark")
sjd = denmark[, c("LRM", "LRY", "IBO", "IDE")]

# use the single lag-order p=2 to determine only the break period #
R.vars  = VAR(sjd, type="both", p=1, season=4)
R.speci = speci.VAR(R.vars, lag_set=2, dim_m=1, trim=3, add_dummy=FALSE)

library("ggfortify")
autoplot(ts(R.speci$df[3:5], start=1+R.speci$args_speci$trim), 
 main="For a single 'p', all IC just reflect the variation of det(UU').")
print(R.speci)

# perform cointegration test procedure with detrending #
R.t_D   = list(t_shift=8, n.season=4)
R.coint = coint.SL(sjd, dim_p=2, type_SL="SL_trend", t_D=R.t_D)
summary(R.coint)

# m=1: line plot #
library("ggplot2")
R.speci1 = speci.VAR(R.vars, lag_set=1:5, dim_m=1, trim=6)
R.values = c("#BDD7E7", "#6BAED6", "#3182BD", "#08519C", "#08306B")
F.line   = ggplot(R.speci1$df) +
  geom_line( aes(x=tau_1, y=HQC, color=as.factor(p), group=as.factor(p))) +
  geom_point(aes(x=tau_1, y=HQC, color=as.factor(p), group=as.factor(p))) +
  geom_point(x=R.speci1$selection["tau_1", "HQC"], 
             y=min(R.speci1$df$HQC), color="red") +
  scale_x_continuous(limits=c(1, nrow(sjd))) +
  scale_color_manual(values=R.values) +
  labs(x=expression(tau), y="HQ Criterion", color="Lag order", title=NULL) +
  theme_bw()
plot(F.line)

# m=2: discrete heat map #
R.speci2 = speci.VAR(R.vars, lag_set=2, dim_m=2, trim=3)
dim_T    = nrow(sjd)  # total sample size
F.heat   = ggplot(R.speci2$df) +
  geom_point(aes(x=tau_1, y=tau_2, color=AIC), size=3) +
  geom_abline(intercept=0, slope=-1, color="grey") +
  scale_x_continuous(limits=c(1, dim_T), expand=c(0, 0)) +
  scale_y_reverse(limits=c(dim_T, 1), expand=c(0, 0)) +
  scale_color_continuous(type="viridis") +
  labs(x=expression(tau[1]), y=expression(tau[2]), color="AIC", title=NULL) +
  theme_bw()
plot(F.heat)

}
\references{
Bai, J., and Perron, P. (1998): 
  "Estimating and Testing Linear Models with Multiple Structural Changes", 
  \emph{Econometrica}, 66, pp. 47-78.

Bai, J., and Perron, P. (2003): 
  "Computation and Analysis of Multiple Structural Change Models", 
  \emph{Journal of Applied Econometrics}, 18, pp. 1-22.

Luetkepohl, H., Saikkonen, P., and Trenkler, C. (2004): 
  "Testing for the Cointegrating Rank of a VAR Process with Level Shift at Unknown Time", 
  \emph{Econometrica}, 72, pp. 647-662.

Yang, M. (2002): 
  "Lag Length and Mean Break in Stationary VAR Models", 
  \emph{Econometrics Journal}, 5, pp. 374-386.
}
\seealso{
Other specification functions: 
\code{\link{speci.factors}()}
}
\concept{specification functions}
