% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sboot.R
\name{sboot.pmb}
\alias{sboot.pmb}
\title{Bootstrap with residual panel blocks for panel SVAR models}
\usage{
sboot.pmb(
  x,
  b.dim = c(1, 1),
  n.ahead = 20,
  n.boot = 500,
  n.cores = 1,
  fix_beta = TRUE,
  deltas = cumprod((100:0)/100),
  normf = NULL,
  w = NULL,
  MG_IRF = TRUE
)
}
\arguments{
\item{x}{Panel VAR object of class '\code{pid}' or '\code{pvarx}' 
or a list of VAR objects that will be \link[=as.varx]{coerced} to '\code{varx}'.
If a list \code{x$L.PSI_bc} of \eqn{N} bias terms are available 
for the \eqn{N} coefficient matrices \eqn{A_i} (such as in \code{sboot2}), 
the bias-corrected second-step of the bootstrap-after-bootstrap procedure 
by Empting et al. (2025) is performed.}

\item{b.dim}{Vector of two integers. The dimensions \eqn{(b_{(t)}, b_{(i)})} 
of each residual panel block for temporal and cross-sectional resampling. The default 
\code{c(1, 1)} specifies an \eqn{i.i.d.} resampling in both dimensions, 
\code{c(1, FALSE)} a temporal resampling, and 
\code{c(FALSE, 1)} a cross-sectional resampling. 
Integers \eqn{> 1} assemble blocks of consecutive residuals.}

\item{n.ahead}{Integer. Number of periods to consider after the initial impulse, i.e. the horizon of the IRF.}

\item{n.boot}{Integer. Number of bootstrap iterations.}

\item{n.cores}{Integer. Number of allocated processor cores.}

\item{fix_beta}{Logical. If \code{TRUE} (the default), the cointegrating vectors \eqn{\beta} 
are fixed over all bootstrap iterations. Ignored in case of rank-unrestricted VAR. 
Use this for VECM with known \eqn{\beta}, too. Note that \eqn{\beta} is fixed in \code{vars:::.bootsvec}, 
but not in \code{vars:::.bootirfsvec} or \code{vars:::.bootirfvec2var}.}

\item{deltas}{Vector. Numeric weights \eqn{\delta_j} that are successively 
multiplied to each bias estimate \eqn{\hat{\Psi}_i} for a stationary correction. 
The default weights \code{deltas = cumprod((100:0)/100)} correspond
to the iterative correction procedure of Step 1b in Kilian (1998). 
Choosing \code{deltas = NULL} deactivates the bootstrap-after-bootstrap procedure.}

\item{normf}{Function. A given function that normalizes the \eqn{K \times S} input-matrix 
into an output matrix of same dimension. See the example in \code{\link{id.iv}} 
for the normalization of Jentsch and Lunsford (2021) 
that fixes the size of the impact response in the IRF.}

\item{w}{Numeric, logical, or character vector. 
\eqn{N} numeric elements weighting the individual coefficients, or 
names or \eqn{N} logical elements selecting a subset from the 
individuals \eqn{i = 1, \ldots, N} for the MG estimation. If \code{NULL} 
(the default), all \eqn{N} individuals are included without weights.}

\item{MG_IRF}{Logical. If \code{TRUE} (the default), the mean-group of individual 
IRF is calculated in accordance with Gambacorta et al. (2014). If \code{FALSE}, 
the IRF is calculated for the mean-group of individual VAR estimates.}
}
\value{
A list of class '\code{sboot2}' with elements:
\item{true}{Mean group estimate of impulse response functions.}
\item{bootstrap}{List of length \code{nboot} holding bootstrap impulse response functions.}
\item{A}{List for the VAR coefficients containing 
  the matrix of point estimates '\code{par}' and 
  the array of bootstrap results '\code{sim}'.}
\item{B}{List for the structural impact matrix containing 
  the matrix of point estimates '\code{par}' and 
  the array of bootstrap results '\code{sim}'.}
\item{L.PSI_bc}{List of the \eqn{N} estimated bias terms \eqn{\hat{\Psi}_i} 
  for the individual VAR coefficients \eqn{\hat{A}_i} according to Kilian (1998).}
\item{pvarx}{Input panel VAR object of class '\code{pvarx}' 
  that has been subjected to the first-step bias-correction.}
\item{b.dim}{Dimensions of each block.}
\item{nboot}{Number of correct bootstrap iterations.}
\item{design}{Character indicating that the recursive design bootstrap has been performed.}
\item{method}{Used bootstrap method.}
\item{stars_t}{Matrix of (\eqn{T \times }\code{n.boot}) integers containing 
  the \eqn{T} temporal resampling draws from each bootstrap iteration.}
\item{stars_i}{Matrix of (\eqn{N \times }\code{n.boot}) integers containing 
  the \eqn{N} cross-sectional resampling draws from each bootstrap iteration.}
}
\description{
Calculates confidence bands for impulse response functions via recursive-design bootstrap.
}
\details{
In case of heterogeneous lag-orders \eqn{p_i} or sample sizes \eqn{T_i},
  the initial periods are fixed in accordance with the usage of presamples. 
  Only the \eqn{(K \times T_{min} \times N)} array of the \eqn{T_{min} = min(T_1,\ldots,T_N)} 
  last residuals is resampled.
}
\examples{
\donttest{
# select minimal or full example #
is_min = TRUE
n.boot = ifelse(is_min, 5, 500)

# prepare data panel #
data("PCAP")
names_k = c("g", "k", "l", "y")  # variable names
names_i = levels(PCAP$id_i)      # country names 
L.data  = sapply(names_i, FUN=function(i) 
  ts(PCAP[PCAP$id_i==i, names_k], start=1960, end=2019, frequency=1), 
  simplify=FALSE)
R.lags = c(2, 4, 2, 3, 2, 4, 4, 2, 2, 3, 3, 3, 2, 4, 4, 2, 2, 2, 4, 2, 2, 2, 4)
names(R.lags) = names_i

# estimate, identify, and bootstrap #
R.pvar = pvarx.VAR(L.data, lags=R.lags, type="both")
R.pid  = pid.chol(R.pvar)
R.boot = sboot.pmb(R.pid, n.boot=n.boot)
summary(R.boot, idx_par="A", level=0.95)  # VAR coefficients with 95\%-confidence intervals
plot(R.boot, lowerq = c(0.05, 0.1, 0.16), upperq = c(0.95, 0.9, 0.84))

# second step of bootstrap-after-bootstrap #
R.bab = sboot.pmb(R.boot, n.boot=n.boot)
summary(R.bab, idx_par="A", level=0.95)  # VAR coefficients with 95\%-confidence intervals
plot(R.bab, lowerq = c(0.05, 0.1, 0.16), upperq = c(0.95, 0.9, 0.84))
}

}
\references{
Brueggemann R., Jentsch, C., and Trenkler, C. (2016): 
  "Inference in VARs with Conditional Heteroskedasticity of Unknown Form", 
  \emph{Journal of Econometrics}, 191, pp. 69-85.

Empting, L. F. T., Maxand, S., Oeztuerk, S., and Wagner, K. (2025): 
  "Inference in Panel SVARs with Cross-Sectional Dependence of Unknown Form".

Kapetanios, G. (2008): 
  "A Bootstrap Procedure for Panel Data Sets with many Cross-sectional Units", 
  \emph{The Econometrics Journal}, 11, pp.377-395.

Kilian, L. (1998): 
  "Small-Sample Confidence Intervals for Impulse Response Functions",
  \emph{Review of Economics and Statistics}, 80, pp. 218-230.

Gambacorta L., Hofmann B., and Peersman G. (2014):
  "The Effectiveness of Unconventional Monetary Policy at the Zero Lower Bound: A Cross-Country Analysis",
  \emph{Journal of Money, Credit and Banking}, 46, pp. 615-642.
}
\seealso{
For the the individual counterpart see \code{\link{sboot.mb}}.
}
