#' Summarize DMR results from \code{mmcmcBayes}
#'
#' @description
#' Convenience function to summarize the DMR table returned by
#' \code{mmcmcBayes()}. It reports the number of detected regions, summaries of
#' region sizes and decision values, and counts by chromosome and (optionally)
#' by stage.
#'
#' @details
#' The function is designed to work with the data frame returned by
#' \code{mmcmcBayes()}. At minimum, the input \code{dmr_table} is expected to
#' contain the columns \code{Chromosome}, \code{CpG_Count}, and
#' \code{Decision_Value}. If a \code{Stage} column is present, counts by stage
#' are also reported.
#'
#' If \code{dmr_table} is \code{NULL} or has zero rows, a summary object with
#' \code{n_dmrs = 0} is returned and all other components are \code{NULL}.
#'
#' @param dmr_table A data frame of DMRs, typically the return value of
#'   \code{mmcmcBayes()}.
#'
#' @return
#' An object of class \code{"mmcmcBayes_dmr_summary"}, which is a list with
#' elements:
#' \itemize{
#'   \item \code{n_dmrs}: Total number of detected DMRs.
#'   \item \code{size_summary}: Named numeric vector of summary statistics for
#'     \code{CpG_Count} (min, 1st quartile, median, mean, 3rd quartile, max),
#'     or \code{NULL} if \code{CpG_Count} is not available.
#'   \item \code{decision_summary}: Named numeric vector of summary statistics
#'     for \code{Decision_Value} (min, 1st quartile, median, mean,
#'     3rd quartile, max), or \code{NULL} if \code{Decision_Value} is not
#'     available.
#'   \item \code{by_chromosome}: A data frame with counts of DMRs per
#'     chromosome, or \code{NULL} if \code{Chromosome} is not available.
#'   \item \code{by_stage}: A data frame with counts of DMRs per stage, if a
#'     \code{Stage} column is present; otherwise \code{NULL}.
#' }
#'
#' @examples
#' \donttest{
#' data(cancer_demo)
#' data(normal_demo)
#'
#' mcmc <- list(nburn = 1000, niter = 2000, thin = 1)
#'
#' set.seed(2021)
#' dmr_res <- mmcmcBayes(
#'   cancer_demo, normal_demo,
#'   stage         = 1,
#'   max_stages    = 2,
#'   num_splits    = 5,
#'   mcmc          = mcmc,
#'   priors_cancer = NULL,
#'   priors_normal = NULL,
#'   bf_thresholds = c(0.5, 0.8, 1.05)
#' )
#'
#' summary_res <- summarize_dmrs(dmr_res)
#' summary_res
#' }
#'
#' @importFrom stats quantile median
#'
#' @export
summarize_dmrs <- function(dmr_table) {
  
  # Handle NULL or empty input gracefully
  if (is.null(dmr_table) || nrow(dmr_table) == 0L) {
    out <- list(
      n_dmrs           = 0L,
      size_summary     = NULL,
      decision_summary = NULL,
      by_chromosome    = NULL,
      by_stage         = NULL
    )
    class(out) <- "mmcmcBayes_dmr_summary"
    return(out)
  }
  
  # Helper to safely summarize a numeric vector
  summarize_numeric <- function(x) {
    x <- x[is.finite(x)]
    if (length(x) == 0L) return(NULL)
    stats <- c(
      min    = min(x),
      q1     = as.numeric(stats::quantile(x, 0.25, names = FALSE)),
      median = stats::median(x),
      mean   = mean(x),
      q3     = as.numeric(stats::quantile(x, 0.75, names = FALSE)),
      max    = max(x)
    )
    stats
  }
  
  # Number of DMRs
  n_dmrs <- nrow(dmr_table)
  
  # Region size summary
  if ("CpG_Count" %in% names(dmr_table)) {
    size_summary <- summarize_numeric(dmr_table$CpG_Count)
  } else {
    size_summary <- NULL
  }
  
  # Decision value summary
  if ("Decision_Value" %in% names(dmr_table)) {
    decision_summary <- summarize_numeric(dmr_table$Decision_Value)
  } else {
    decision_summary <- NULL
  }
  
  # Counts by chromosome
  if ("Chromosome" %in% names(dmr_table)) {
    tab_chr <- table(dmr_table$Chromosome, useNA = "no")
    by_chromosome <- data.frame(
      Chromosome = names(tab_chr),
      n_dmrs     = as.integer(tab_chr),
      row.names  = NULL,
      check.names = FALSE
    )
  } else {
    by_chromosome <- NULL
  }
  
  # Counts by stage (if available)
  if ("Stage" %in% names(dmr_table)) {
    tab_stage <- table(dmr_table$Stage, useNA = "no")
    by_stage <- data.frame(
      Stage  = names(tab_stage),
      n_dmrs = as.integer(tab_stage),
      row.names   = NULL,
      check.names = FALSE
    )
  } else {
    by_stage <- NULL
  }
  
  out <- list(
    n_dmrs           = n_dmrs,
    size_summary     = size_summary,
    decision_summary = decision_summary,
    by_chromosome    = by_chromosome,
    by_stage         = by_stage
  )
  
  class(out) <- "mmcmcBayes_dmr_summary"
  out
}

#' @export
print.mmcmcBayes_dmr_summary <- function(x, ...) {
  cat("Summary of DMR results (mmcmcBayes)\n")
  cat("-----------------------------------\n")
  cat("Number of DMRs:", x$n_dmrs, "\n\n")
  
  if (!is.null(x$size_summary)) {
    cat("Region size (CpG_Count):\n")
    print(round(x$size_summary, 3))
    cat("\n")
  } else {
    cat("Region size (CpG_Count): not available\n\n")
  }
  
  if (!is.null(x$decision_summary)) {
    cat("Decision_Value:\n")
    print(round(x$decision_summary, 3))
    cat("\n")
  } else {
    cat("Decision_Value: not available\n\n")
  }
  
  if (!is.null(x$by_chromosome)) {
    cat("DMRs by Chromosome:\n")
    print(x$by_chromosome)
    cat("\n")
  }
  
  if (!is.null(x$by_stage)) {
    cat("DMRs by Stage:\n")
    print(x$by_stage)
    cat("\n")
  }
  
  invisible(x)
}
