% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lsirmgrm2pl.R
\name{lsirmgrm2pl}
\alias{lsirmgrm2pl}
\title{Fit an ordinal LSIRM with the graded response model (2PL)}
\usage{
lsirmgrm2pl(
  data,
  ncat = NULL,
  missing_data = NA,
  missing.val = 99,
  chains = 1,
  multicore = 1,
  seed = NA,
  ndim = 2,
  niter = 15000,
  nburn = 2500,
  nthin = 5,
  nprint = 500,
  jump_beta = 0.4,
  jump_theta = 1,
  jump_alpha = 1,
  jump_gamma = 0.2,
  jump_z = 0.5,
  jump_w = 0.5,
  pr_mean_beta = 0,
  pr_sd_beta = 1,
  pr_mean_theta = 0,
  pr_sd_theta = 1,
  pr_mean_alpha = 0.5,
  pr_sd_alpha = 1,
  pr_mean_gamma = 0.5,
  pr_sd_gamma = 1,
  pr_a_theta = 0.001,
  pr_b_theta = 0.001,
  fixed_gamma = FALSE,
  spikenslab = FALSE,
  pr_spike_mean = -3,
  pr_spike_sd = 1,
  pr_slab_mean = 0.5,
  pr_slab_sd = 1,
  pr_xi_a = 1,
  pr_xi_b = 1,
  adapt = NULL,
  verbose = FALSE,
  fix_theta_sd = FALSE,
  fix_alpha_1 = TRUE
)
}
\arguments{
\item{data}{Matrix; an ordinal (ordered categorical) item response matrix. Each row represents a respondent, and
each column represents an item. Values can be either \code{0:(K-1)} or \code{1:K}. Missing values can be \code{NA}.}

\item{ncat}{Integer; number of categories \eqn{K}. If \code{NULL}, it is inferred from the observed data.}

\item{missing_data}{Character; the type of missing data assumed. Options are \code{NA}, \code{"mar"}, or
\code{"mcar"}. If \code{NA} and \code{data} contains missing values, it is set to \code{"mcar"} internally.}

\item{missing.val}{Numeric; numeric code used to represent missing values in the C++ sampler. Default is 99.}

\item{chains}{Integer; number of MCMC chains. Default is 1.}

\item{multicore}{Integer; number of cores for parallel execution when \code{chains > 1}. Default is 1.}

\item{seed}{Integer; RNG seed. Default is \code{NA}.}

\item{ndim}{Integer; latent space dimension. Default is 2.}

\item{niter}{Integer; total MCMC iterations. Default is 15000.}

\item{nburn}{Integer; burn-in iterations. Default is 2500.}

\item{nthin}{Integer; thinning interval. Default is 5.}

\item{nprint}{Integer; print interval if \code{verbose=TRUE}. Default is 500.}

\item{jump_beta}{Numeric; proposal SD for GRM thresholds. Default is 0.4. During MCMC sampling, threshold proposals are constrained to maintain the ordering \eqn{\beta_{i,1} > \beta_{i,2} > \cdots > \beta_{i,K-1}} for each item.}

\item{jump_theta}{Numeric; proposal SD for theta. Default is 1.}

\item{jump_alpha}{Numeric; proposal SD on log-scale for \eqn{\alpha}. Default is 1.}

\item{jump_gamma}{Numeric; proposal SD on log-scale for gamma. Default is 0.2.}

\item{jump_z}{Numeric; proposal SD for z. Default is 0.5.}

\item{jump_w}{Numeric; proposal SD for w. Default is 0.5.}

\item{pr_mean_beta}{Numeric; prior mean for thresholds. Default is 0.}

\item{pr_sd_beta}{Numeric; prior SD for thresholds. Default is 1.}

\item{pr_mean_theta}{Numeric; prior mean for theta. Default is 0.}

\item{pr_sd_theta}{Numeric; prior SD for theta. Default is 1.}

\item{pr_mean_alpha}{Numeric; log-normal prior mean for \eqn{\alpha}. Default is 0.5.}

\item{pr_sd_alpha}{Numeric; log-normal prior SD for \eqn{\alpha}. Default is 1.}

\item{pr_mean_gamma}{Numeric; log-normal prior mean for gamma. Default is 0.5.}

\item{pr_sd_gamma}{Numeric; log-normal prior SD for gamma. Default is 1.}

\item{pr_a_theta}{Numeric; shape for inverse-gamma prior on var(theta). Default is 0.001.}

\item{pr_b_theta}{Numeric; scale for inverse-gamma prior on var(theta). Default is 0.001.}

\item{fixed_gamma}{Logical; if TRUE, fixes \eqn{\gamma = 1} (no sampling). Default is FALSE.}

\item{spikenslab}{Logical; if TRUE, uses spike-and-slab priors for \eqn{\gamma}. Default is FALSE.}

\item{pr_spike_mean}{Numeric; prior mean for the spike component (on log-scale). Default is -3.}

\item{pr_spike_sd}{Numeric; prior SD for the spike component (on log-scale). Default is 1.}

\item{pr_slab_mean}{Numeric; prior mean for the slab component (on log-scale). Default is 0.5.}

\item{pr_slab_sd}{Numeric; prior SD for the slab component (on log-scale). Default is 1.}

\item{pr_xi_a}{Numeric; Beta prior shape a for mixing weight \eqn{\xi}. Default is 1.}

\item{pr_xi_b}{Numeric; Beta prior shape b for mixing weight \eqn{\xi}. Default is 1.}

\item{adapt}{List; optional adaptive MCMC control. If not \code{NULL}, proposal standard deviations are adapted during the burn-in period to reach a target acceptance rate and are held fixed during the main MCMC sampling.
When adaptation is enabled, the reported acceptance ratios in the output (\code{accept_beta}, \code{accept_theta}, \code{accept_alpha}, etc.) are computed only from iterations after burn-in, reflecting the performance of the adapted proposal distributions.
Elements of the list can include:
\itemize{
  \item \code{use_adapt}: Logical; if \code{TRUE}, adaptive MCMC is used. Default is \code{FALSE}.
  \item \code{adapt_interval}: Integer; the number of iterations between each update of the proposal SDs. Default is \code{100}.
  \item \code{adapt_rate}: Numeric; Robbins-Monro scaling constant (c) in step size formula: adapt_rate / iteration^decay_rate. Default is \code{1.0}. Valid range: any positive value. Recommended: 0.5-2.0.
  \item \code{decay_rate}: Numeric; Robbins-Monro decay exponent (alpha) in step size formula. Default is \code{0.5}. Valid range: (0.5, 1]. Recommended: 0.5-0.8.
  \item \code{target_accept}: Numeric; target acceptance rate for scalar parameters (beta, theta, gamma, alpha). Default is \code{0.44}.
  \item \code{target_accept_zw}: Numeric; target acceptance rate for multi-dimensional latent positions z and w. Default is \code{0.234}.
  \item \code{target_accept_beta/theta/alpha/gamma}: Numeric; (optional) parameter-specific target acceptance rates to override \code{target_accept}.
}}

\item{verbose}{Logical; If TRUE, MCMC progress and parameter samples are printed to the console during execution. Default is FALSE.}

\item{fix_theta_sd}{Logical; If TRUE, the standard deviation of the respondent latent positions \eqn{\theta} is fixed at 1 instead of being sampled. Default is FALSE.}

\item{fix_alpha_1}{Logical; if TRUE, fixes \eqn{\alpha_1 = 1}. Default is TRUE.}
}
\value{
An object of class \code{lsirm}. For multi-chain fits, a list where each element (\code{chain1}, \code{chain2}, etc.) is a single-chain fit of class \code{lsirm}.

If \code{missing_data = "mar"}, the returned object additionally contains \code{imp} (MCMC draws of imputed
responses for each missing cell) and \code{imp_estimate} (posterior mean imputation for each missing cell).
}
\description{
\code{\link{lsirmgrm2pl}} fits a two-parameter logistic (2PL) extension of the ordinal latent space item response model
using the graded response model (GRM) with item discrimination parameters \eqn{\alpha_i}.
The model factorizes the item response matrix into item thresholds, respondent ability, and item discrimination,
while embedding interaction effects in a latent space. The resulting latent space provides an interaction map
that visualizes the complex relationships between respondents and items beyond traditional IRT models.
}
\details{
\code{lsirmgrm2pl} implements the 2PL extension of the Graded Response Model (GRM) in a latent space framework.
Let \eqn{Y_{j,i} \in \{0,\ldots,K-1\}} be the ordered categorical response of respondent \eqn{j} to item \eqn{i}.
The model is defined via cumulative logits:
\deqn{\Pr(Y_{j,i} \ge k | \theta_j, \alpha_i, \beta_{i,k}, \gamma, z_j, w_i) = \text{logit}^{-1}(\alpha_i \theta_j + \beta_{i,k} - \gamma\,\|z_j-w_i\|)}
for \eqn{k=1,\ldots,K-1}, where \eqn{\alpha_i} is the item discrimination parameter and \eqn{\beta_{i,k}} are item-specific thresholds that satisfy the ordering constraint \eqn{\beta_{i,1} > \beta_{i,2} > \cdots > \beta_{i,K-1}} for identifiability.

Missing data handling:
\itemize{
  \item \code{"mcar"}: missing responses are excluded from the likelihood.
  \item \code{"mar"}: missing responses are imputed by data augmentation within the MCMC.
}
}
\examples{
# Generate example ordinal item response matrix
set.seed(123)
nsample <- 50
nitem <- 10
data <- matrix(sample(1:5, nsample * nitem, replace = TRUE), nrow = nsample)

# Fit 2PL GRM LSIRM using direct function call
fit <- lsirmgrm2pl(data, niter = 1000, nburn = 500, nthin = 2)
summary(fit)

# Fit with missing data (MAR)
data_mar <- data
data_mar[sample(1:length(data), 20)] <- NA
fit_mar <- lsirm(data_mar ~ lsirmgrm2pl(missing_data = "mar", niter = 1000, nburn = 500))

# Fit with Spike-and-Slab prior for model selection
fit_ss <- lsirm(data ~ lsirmgrm2pl(spikenslab = TRUE, niter = 1000, nburn = 500))

# Fit with adaptive MCMC for automatic tuning
fit_adapt <- lsirmgrm2pl(data, niter = 2000, nburn = 1000,
                         adapt = list(use_adapt = TRUE, adapt_interval = 50))
# Check adapted jump sizes and acceptance rates
cat("Final jump_alpha:", fit_adapt$jump_alpha, "\n")
cat("Acceptance rate (post-burnin):", fit_adapt$accept_alpha, "\n")

}
\references{
De Carolis, L., Kang, I., & Jeon, M. (2025). A Latent Space Graded Response Model for Likert-Scale
Psychological Assessments. \emph{Multivariate Behavioral Research}. \doi{10.1080/00273171.2025.2605678}
}
