% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensemakr.R
\name{sensemakr}
\alias{sensemakr}
\alias{sensemakr.iv_fit}
\title{Sensitivity Analysis of Instrumental Variable Estimates}
\usage{
sensemakr(...)

\method{sensemakr}{iv_fit}(
  model,
  benchmark_covariates = NULL,
  kz = 1,
  ky = kz,
  kd = kz,
  r2zw.x = NULL,
  r2y0w.zx = r2zw.x,
  bound_label = "Manual Bound",
  q = 1,
  alpha = 0.05,
  min = TRUE,
  ...
)
}
\arguments{
\item{...}{arguments passed to other methods.}

\item{model}{a model created with the function \code{\link{iv_fit}}.}

\item{benchmark_covariates}{character vector of the names of covariates that will be used to bound the plausible strength of the latent variables.}

\item{kz}{numeric vector. Parameterizes how many times stronger the latent variables are related to the instrument in comparison to the observed benchmark covariates.
Default value is \code{1} (latent variable is as strong as benchmark covariate).}

\item{ky}{numeric vector. Parameterizes how many times stronger the latent variables are related to the (pot.) outcome in comparison to the observed benchmark covariates.}

\item{kd}{numeric vector. Parameterizes how many times stronger the latent variables are related to the treatment in comparison to the observed benchmark covariates. Default value is the same as \code{kz}.}

\item{r2zw.x}{(optional) hypothetical partial R2 of latent variables
W with the instrument Z, given observed covariates X.}

\item{r2y0w.zx}{(optional) hypothetical partial R2 of latent variables W with the (pot.) outcome Y(0) given Z and X. Default is the same as \code{r2zw.x}.}

\item{bound_label}{label to bounds provided manually in \code{r2zw.x} and \code{r2y0w.zx}.}

\item{q}{percent change of the effect estimate that would be deemed problematic. Default is 1, which means a reduction of 100\% of the current effect estimate (bring estimate to zero).}

\item{alpha}{significance level.}

\item{min}{should we consider biases as large or larger than a certain amount? Default is \code{TRUE}.}
}
\value{
An object of class \code{iv.sensemakr}, containing:
\describe{
 \item{ \code{pars} }{A \code{list} with the general parameters used when calling sensemakr.}
 \item{ \code{unadjusted} }{A \code{list} with the original, unadjusted results.}
 \item{ \code{sensitivity_stats} }{A \code{list} with the sensitivity statistics of the IV, First-Stage, and Reduced-Form regressions.}
 \item{ \code{bounds} }{A \code{list} with bounds on the strength of latent variables if they were "k times" as strong as the benchmark covariates.}
 }
}
\description{
This function performs sensitivity analysis of instrumental variable estimates, as discussed in Cinelli and Hazlett (2025).
The main input is an object of class \code{\link{iv_fit}}. It returns an object of class \code{iv.sensemakr} with several pre-computed sensitivity statistics for reporting. After running \code{sensemakr} you may directly use the \code{plot}, \code{print} and \code{summary} methods in the returned object.
}
\examples{

# loads package
library(iv.sensemakr)

# loads dataset
data("card")

# prepares data
y <- card$lwage  # outcome
d <- card$educ   # treatment
z <- card$nearc4 # instrument
x <- model.matrix( ~ exper + expersq + black + south + smsa + reg661 + reg662 +
                     reg663 + reg664 + reg665+ reg666 + reg667 + reg668 + smsa66,
                   data = card) # covariates
# fits IV model
card.fit <- iv_fit(y,d,z,x)

# see results
card.fit

# runs sensitivity analysis
card.sens <- sensemakr(card.fit, benchmark_covariates = c("black", "smsa"))

# see results
card.sens

# sensitivity contour plot
plot(card.sens, lim = 0.09)
}
\references{
Cinelli, C. and Hazlett, C. (2025), "An Omitted Variable Bias Framework for Sensitivity Analysis of Instrumental Variables." Biometrika. \doi{10.1093/biomet/asaf004}
}
