/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Tpkg.h"      

static herr_t H5T__pack(const H5T_t *dt);
static htri_t H5T__is_packed(const H5T_t *dt);
static H5T_t *H5T__reopen_member_type(const H5T_t *dt, unsigned membno);

size_t
H5Tget_member_offset(hid_t type_id, unsigned membno)
{
    H5T_t *dt;        
    size_t ret_value; 

    FUNC_ENTER_API(0)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)) || H5T_COMPOUND != dt->shared->type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "not a compound datatype");
    if (membno >= dt->shared->u.compnd.nmembs)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid member number");

    
    ret_value = H5T_GET_MEMBER_OFFSET(dt->shared, membno);

done:
    FUNC_LEAVE_API(ret_value)
} 

size_t
H5T_get_member_offset(const H5T_t *dt, unsigned membno)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(dt);
    assert(membno < dt->shared->u.compnd.nmembs);

    FUNC_LEAVE_NOAPI(dt->shared->u.compnd.memb[membno].offset)
} 

H5T_class_t
H5Tget_member_class(hid_t type_id, unsigned membno)
{
    H5T_t      *dt;        
    H5T_class_t ret_value; 

    FUNC_ENTER_API(H5T_NO_CLASS)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)) || H5T_COMPOUND != dt->shared->type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_NO_CLASS, "not a compound datatype");
    if (membno >= dt->shared->u.compnd.nmembs)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5T_NO_CLASS, "invalid member number");

    
    ret_value = H5T_GET_CLASS(dt->shared->u.compnd.memb[membno].type->shared, false);

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Tget_member_type(hid_t type_id, unsigned membno)
{
    H5T_t *dt;             
    H5T_t *memb_dt = NULL; 
    hid_t  ret_value;      

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)) || H5T_COMPOUND != dt->shared->type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a compound datatype");
    if (membno >= dt->shared->u.compnd.nmembs)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid member number");

    
    if (NULL == (memb_dt = H5T__reopen_member_type(dt, membno)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID, "unable to retrieve member type");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, memb_dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable register datatype ID");

done:
    if (ret_value < 0)
        if (memb_dt && H5T_close(memb_dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, H5I_INVALID_HID, "can't close datatype");

    FUNC_LEAVE_API(ret_value)
} 

H5T_t *
H5T_get_member_type(const H5T_t *dt, unsigned membno)
{
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(dt);
    assert(membno < dt->shared->u.compnd.nmembs);

    
    if (NULL == (ret_value = H5T_copy(dt->shared->u.compnd.memb[membno].type, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "unable to copy member datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__reopen_member_type(const H5T_t *dt, unsigned membno)
{
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(membno < dt->shared->u.compnd.nmembs);

    
    if (NULL == (ret_value = H5T_copy_reopen(dt->shared->u.compnd.memb[membno].type)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "unable to reopen member datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5T__get_member_size(const H5T_t *dt, unsigned membno)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);
    assert(membno < dt->shared->u.compnd.nmembs);

    FUNC_LEAVE_NOAPI(dt->shared->u.compnd.memb[membno].type->shared->size)
} 

herr_t
H5Tinsert(hid_t parent_id, const char *name, size_t offset, hid_t member_id)
{
    H5T_t *parent;              
    H5T_t *member;              
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (parent_id == member_id)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "can't insert compound datatype within itself");
    if (NULL == (parent = (H5T_t *)H5I_object_verify(parent_id, H5I_DATATYPE)) ||
        H5T_COMPOUND != parent->shared->type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a compound datatype");
    if (H5T_STATE_TRANSIENT != parent->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "parent type read-only");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no member name");
    if (NULL == (member = (H5T_t *)H5I_object_verify(member_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

    
    if (H5T__insert(parent, name, offset, member) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "unable to insert member");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tpack(hid_t type_id)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)) ||
        H5T_detect_class(dt, H5T_COMPOUND, true) <= 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a compound datatype");

    
    if (H5T__pack(dt) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to pack compound datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5T__insert(H5T_t *parent, const char *name, size_t offset, const H5T_t *member)
{
    unsigned idx; 
    size_t   total_size;
    unsigned i;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(parent && H5T_COMPOUND == parent->shared->type);
    assert(H5T_STATE_TRANSIENT == parent->shared->state);
    assert(member);
    assert(name && *name);

    
    for (i = 0; i < parent->shared->u.compnd.nmembs; i++)
        if (!strcmp(parent->shared->u.compnd.memb[i].name, name))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "member name is not unique");

    
    total_size = member->shared->size;
    for (i = 0; i < parent->shared->u.compnd.nmembs; i++)
        if ((offset <= parent->shared->u.compnd.memb[i].offset &&
             (offset + total_size) > parent->shared->u.compnd.memb[i].offset) ||
            (parent->shared->u.compnd.memb[i].offset <= offset &&
             (parent->shared->u.compnd.memb[i].offset + parent->shared->u.compnd.memb[i].size) > offset))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "member overlaps with another member");

    
    if ((offset + total_size) > parent->shared->size)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "member extends past end of compound type");

    
    if (parent->shared->u.compnd.nmembs >= parent->shared->u.compnd.nalloc) {
        unsigned     na = MAX(1, parent->shared->u.compnd.nalloc * 2);
        H5T_cmemb_t *x = (H5T_cmemb_t *)H5MM_realloc(parent->shared->u.compnd.memb, na * sizeof(H5T_cmemb_t));

        if (!x)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed");
        parent->shared->u.compnd.nalloc = na;
        parent->shared->u.compnd.memb   = x;
    } 

    
    idx                                       = parent->shared->u.compnd.nmembs;
    parent->shared->u.compnd.memb[idx].offset = offset;
    parent->shared->u.compnd.memb[idx].size   = total_size;
    if (NULL == (parent->shared->u.compnd.memb[idx].name = H5MM_xstrdup(name)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't duplicate name string");
    if (NULL == (parent->shared->u.compnd.memb[idx].type = H5T_copy(member, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "couldn't copy datatype");

    parent->shared->u.compnd.sorted = H5T_SORT_NONE;
    parent->shared->u.compnd.nmembs++;
    parent->shared->u.compnd.memb_size += total_size;

    
    assert(!(parent->shared->u.compnd.packed));

    
    H5T__update_packed(parent);

    
    if (member->shared->force_conv == true)
        parent->shared->force_conv = true;

    
    if (parent->shared->version < member->shared->version)
        
        
        if (H5T__upgrade_version(parent, member->shared->version) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't upgrade member encoding version");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__pack(const H5T_t *dt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dt);

    if (H5T_detect_class(dt, H5T_COMPOUND, false) > 0) {
        
        if (true == H5T__is_packed(dt))
            HGOTO_DONE(SUCCEED);

        
        if (H5T_STATE_TRANSIENT != dt->shared->state)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "datatype is read-only");

        if (dt->shared->parent) {
            if (H5T__pack(dt->shared->parent) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to pack parent of datatype");

            
            if (dt->shared->type == H5T_ARRAY)
                dt->shared->size = dt->shared->parent->shared->size * dt->shared->u.array.nelem;
            else if (dt->shared->type == H5T_COMPLEX)
                dt->shared->size = 2 * dt->shared->parent->shared->size;
            else if (dt->shared->type != H5T_VLEN)
                dt->shared->size = dt->shared->parent->shared->size;
        } 
        else if (dt->shared->type == H5T_COMPOUND) {
            size_t   offset; 
            unsigned i;      

            
            for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
                if (H5T__pack(dt->shared->u.compnd.memb[i].type) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL,
                                "unable to pack part of a compound datatype");

                
                dt->shared->u.compnd.memb[i].size = (dt->shared->u.compnd.memb[i].type)->shared->size;
            } 

            
            if (H5T__sort_value(dt, NULL) < 0)
                HGOTO_ERROR(H5E_INTERNAL, H5E_CANTCOMPARE, FAIL, "value sort failed");
            for (i = 0, offset = 0; i < dt->shared->u.compnd.nmembs; i++) {
                dt->shared->u.compnd.memb[i].offset = offset;
                offset += dt->shared->u.compnd.memb[i].size;
            }

            
            dt->shared->size = MAX(1, offset);

            
            dt->shared->u.compnd.packed = true;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5T__is_packed(const H5T_t *dt)
{
    htri_t ret_value = true; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);

    
    while (dt->shared->parent)
        dt = dt->shared->parent;

    
    if (dt->shared->type == H5T_COMPOUND)
        ret_value = (htri_t)(dt->shared->u.compnd.packed);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5T__update_packed(const H5T_t *dt)
{
    unsigned i; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);
    assert(dt->shared->type == H5T_COMPOUND);

    
    if (dt->shared->size == dt->shared->u.compnd.memb_size) {
        
        dt->shared->u.compnd.packed = true;

        
        for (i = 0; i < dt->shared->u.compnd.nmembs; i++)
            if (!H5T__is_packed(dt->shared->u.compnd.memb[i].type)) {
                dt->shared->u.compnd.packed = false;
                break;
            } 
    }         
    else
        dt->shared->u.compnd.packed = false;

    FUNC_LEAVE_NOAPI_VOID
} 
