#' @title O'Brien-Flemming error spending function.
#' @description
#' Computes the cumulative Type I error spending function corresponding to the two-sided O'Brien-Flemming group
#' sequential design.
#'
#' @param t Mumeric vector of information fractions, typically in \eqn{(0,1)},
#' where \code{t=1} corresponds to the final analysis.
#' @param alpha Overall two-sided Type I error level. Default is \code{0.05}.
#' @importFrom stats pnorm qnorm
#' @returns A numeric vector giving the cumulative type I error spent at each information fraction \code{t}
#' under the O'Brien-Flemming spending function.
#'
#' @references Jennison, C. and Turnbull, B. W. (2000). \emph{Group Sequential Methods with Applications to Clinical Trials}. Chapman and Hall/CRC.
#' @export
#'
#' @examples
#' # Two-interim analyses at 50% and 75% maximum information and a final analysis
#' OBF(t = c(0.5, 0.7, 1))
OBF <- function(t, alpha = 0.05){
  #two-sided testing
  f_t = pmin(alpha, 4 - 4*pnorm(qnorm(1 - alpha/4)/sqrt(t)))
  return(f_t)
}
