#' Export Results to Excel
#'
#' @param results List containing analysis results.
#' @param output_path Path for the Excel file.
#' @param verbose Logical; print progress messages.
#'
#' @return Invisible path to created file.
#' @export
export_results_xlsx <- function(results, output_path, verbose = TRUE) {
  
  if (!requireNamespace("openxlsx", quietly = TRUE)) {
    stop("Package 'openxlsx' required. Install with install.packages('openxlsx')")
  }
  
  wb <- openxlsx::createWorkbook()
  
  write_sheet <- function(wb, sheet_name, df) {
    if (is.null(df) || !is.data.frame(df) || nrow(df) == 0) {
      df <- data.frame(info = "No data", stringsAsFactors = FALSE)
    }
    openxlsx::addWorksheet(wb, sheet_name)
    openxlsx::writeDataTable(wb, sheet_name, df)
    openxlsx::freezePane(wb, sheet_name, firstActiveRow = 2)
    openxlsx::setColWidths(wb, sheet_name, cols = seq_len(ncol(df)), widths = "auto")
  }
  
  if (!is.null(results$hurdle)) write_sheet(wb, "Hurdle", results$hurdle)
  if (!is.null(results$te)) write_sheet(wb, "TE", results$te)
  if (!is.null(results$placebo)) write_sheet(wb, "Placebo", results$placebo)
  if (!is.null(results$tvarstar)) write_sheet(wb, "TVARSTAR", results$tvarstar)
  if (!is.null(results$varx)) write_sheet(wb, "VARX", results$varx)
  
  openxlsx::saveWorkbook(wb, output_path, overwrite = TRUE)
  
  if (verbose) message("Excel saved to: ", output_path)
  invisible(output_path)
}