% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/image_med.R
\name{imagemed}
\alias{imagemed}
\title{Wasserstein Median of Images}
\usage{
imagemed(images, weights = NULL, C = NULL, ...)
}
\arguments{
\item{images}{a length-\eqn{N} list of same-size grayscale image matrices of size \eqn{(m\times n)}.}

\item{weights}{a weight of each image; if \code{NULL} (default), uniform weight is set.
Otherwise, it should be a length-\eqn{N} vector of nonnegative weights.}

\item{C}{an optional \eqn{(mn\times mn)} ground cost matrix (squared distances). If \code{NULL}
(default), the squared Euclidean grid cost is used.}

\item{...}{extra parameters including \describe{
\item{maxiter}{maximum number of IRLS outer iterations (default: \code{30}).}
\item{abstol}{stopping tolerance based on \eqn{\ell_2} change of iterates (default: \code{1e-6}).}
\item{delta}{small positive number to avoid division by zero in IRLS weights
  (default: \code{1e-8}).}
\item{init.image}{initial median iterate (default: unweighted barycenter via \code{imagebary}
  with a small number of iterations).}
\item{init.bary.iter}{iterations for the default initialization barycenter (default: \code{10}).}
\item{bary.maxiter}{maximum iterations for each barycenter subproblem (default: \code{200}).}
\item{bary.abstol}{tolerance for each barycenter subproblem (default: \code{1e-7}).}
\item{bary.step0}{initial step size for barycenter subproblem (default: \code{0.5}).}
\item{bary.stepschedule}{\code{"sqrt"} or \code{"const"} for barycenter subproblem (default: \code{"sqrt"}).}
\item{bary.eps}{positivity floor used inside barycenter (default: \code{1e-15}).}
\item{bary.smooth}{smoothing used inside barycenter (default: \code{1e-12}).}
\item{bary.clip}{gradient clipping used inside barycenter (default: \code{50}).}
\item{bary.max_backtrack}{backtracking cap used inside barycenter (default: \code{8}).}
\item{print.progress}{logical; if \code{TRUE}, print iteration diagnostics (default: \code{FALSE}).}
}}
}
\value{
an \eqn{(m\times n)} matrix of the median.
}
\description{
Using exact balanced optimal transport as a subroutine, \code{imagemed}
computes an unregularized 2-Wasserstein geometric median image \eqn{X^\dagger}
from multiple input images \eqn{X_1,\ldots,X_N}. The Wasserstein median is
defined as a minimizer of the (weighted) sum of Wasserstein distances,
\deqn{ \arg\min_{X} \sum_{i=1}^N w_i\, W_2(X, X_i). }
}
\details{
Unlike Wasserstein barycenters (which minimize squared distances), the median
is a robust notion of centrality. This function solves the problem with an
iterative reweighted least squares (IRLS) scheme (a Wasserstein analogue of
Weiszfeld's algorithm). Each outer iteration updates weights based on current
distances and then solves a weighted Wasserstein barycenter problem:
\deqn{ \alpha_i^{(k)} \propto \frac{w_i}{\max(W_2(X^{(k)},X_i),\delta)}, \qquad
       X^{(k+1)} = \arg\min_X \sum_{i=1}^N \alpha_i^{(k)}\, W_2^2(X, X_i). }

The barycenter subproblem is solved by \code{\link{imagebary}} (mirror descent
with exact OT dual subgradients). Distances \eqn{W_2} are computed by exact
EMD plans under the same squared ground cost.
}
\examples{
\dontrun{
#----------------------------------------------------------------------
#                             MNIST Example
#
# Use 6 images from digit '8' and 4 images from digit '1'.
# The median should look closer to the shape of '8'.
#----------------------------------------------------------------------
# DATA PREP
set.seed(11)
data(digits)
dat_8 = digits$image[sample(which(digits$label==8), 6)]
dat_1 = digits$image[sample(which(digits$label==1), 4)]
dat_all = c(dat_8, dat_1)

# COMPUTE BARYCENTER AND MEDIAN
img_bary = imagebary(dat_all, maxiter=50)
img_med  = imagemed(dat_all, maxiter=50)

# VISUALIZE
opar <- par(no.readonly=TRUE)
par(mfrow=c(1,2), pty="s")
image(img_bary, axes=FALSE, main="Barycenter")
image(img_med,  axes=FALSE, main="Median")
par(opar)
}

}
\concept{image}
