#' Depict images with masking layers
#'
#' \code{showImgMask} - Depicts an image, e.g., a root scan, with an optional
#' masking layer using a specified color.
#'
#' @param root_img Root image. Can be a PNG, i.e., an array with 3 dimensions
#' (3 layers each containing a 2-dim. numeric matrix with values between
#' 0 and 1), or a 2-dim. matrix. If the root image is NULL (default), then a
#' completely black image is used.
#' @param mask 2-dim. true/false-matrix with the same number of rows and columns
#' as \code{skel_img} (optional, default = NULL, interpreted as a matrix
#' consisting only of TRUEs, i.e., nothing is "removed" from the image).
#' @param mask_col Color of the FALSE-regions of the mask (default "grey").
#' Can be of any of the three kinds of R color specifications, i.e., either a
#' color name (as listed by colors()), a hexadecimal string (see Details), or a
#' positive integer (indicates using palette()).
#'
#' @return \code{showImgMask} No return value, called for side effects
#' (plotting).
#'
#' @export
#' @rdname showImgMask
#'
#' @examples
#' # Basic example:
#' showImgMask(matrix(c(1,0,0,0,
#'                      0,1,0,0,
#'                      0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'              matrix(c(TRUE, FALSE,FALSE,TRUE,
#'                       TRUE, TRUE, TRUE, TRUE,
#'                       FALSE,TRUE, TRUE, FALSE), ncol = 4, nrow = 3,
#'                       byrow = TRUE))
#' # Example using the createDepthLayerMasks()-function:.
#' showImgMask(root_img = matrix(c(1,0,0,0,0,0,0,
#'                       0,1,0,0,0,0,0,
#'                       0,0,1,1,0,0,0), ncol = 7, nrow = 3, byrow = TRUE),
#'              mask = createDepthLayerMasks(ppcm = 1,
#'                           dims_px = c(3,7),
#'                           depth_levels_cm = rbind(c(-1,-2), c(-2,-3)))[[1]],
#'              mask_col ="brown")
showImgMask <- function(root_img = NULL, mask = NULL, mask_col = "grey"){
  root_img <- applyImgMask(root_img, mask, mask_col)
  # Plot the image.
  grid::grid.raster(root_img, interpolate = FALSE)
}
#' Depict images with masking layers
#'
#' \code{applyImgMask} - Changes the colouring of an image, e.g., a root scan,
#' according to a masking layer using a specified color. If only a matrix is
#' provided, it is treated like the r-layer (referring to the rgb color values).
#'
#' @return \code{applyImgMask} An array with 3 dimensions
#' (3 layers each containing a 2-dim. numeric matrix with values between
#' 0 and 1) like a PNG.
#'
#' @export
#' @rdname showImgMask
#'
#' @examples
#' # Basic example:
#' applyImgMask(matrix(c(1,0,0,0,
#'                      0,1,0,0,
#'                      0,0,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'              matrix(c(TRUE, FALSE,FALSE,TRUE,
#'                       TRUE, TRUE, TRUE, TRUE,
#'                       FALSE,TRUE, TRUE, FALSE), ncol = 4, nrow = 3,
#'                       byrow = TRUE))
applyImgMask <- function(root_img = NULL, mask = NULL, mask_col = "grey"){
  # Get the RGB code of the color.
  mask_rgb <- grDevices::col2rgb(mask_col, alpha = FALSE)/255
  # Give the image three layers if it is only a matrix.
  if(is.null(root_img)){
    if(!is.null(mask)){
      root_img <- array(0,
                        dim = c(dim(mask),3))
    } else {
      return(NULL)
    }
  }
  if(length(dim(root_img))==2){
    root_img <- array(c(root_img, root_img, root_img),
                      dim = c(dim(root_img),3))
  }
  # Apply the mask, i.e., color all FALSE pixels.
  # Give the image three layers if it is only a matrix.
  if(!is.null(mask)){
    for(i in 1:3){
      root_img[,,i][!mask] <- mask_rgb[i,1]
    }
  }
  return(root_img)
}
