% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeCorrLoadings.R
\name{makeCorrLoadings}
\alias{makeCorrLoadings}
\title{Generate Inter-Item Correlation Matrix from Factor Loadings}
\usage{
makeCorrLoadings(
  loadings,
  factorCor = NULL,
  uniquenesses = NULL,
  nearPD = FALSE,
  diagnostics = FALSE
)
}
\arguments{
\item{loadings}{Numeric matrix. A \eqn{k \times f} matrix of standardized
factor loadings \eqn{items \times factors}.
Row names and column names are used for diagnostics if present.}

\item{factorCor}{Optional \eqn{f \times f} matrix of factor
correlations (\eqn{\Phi}).
If NULL, assumes orthogonal factors.}

\item{uniquenesses}{Optional vector of length k. If NULL, calculated
as \eqn{1 - rowSums(loadings^2)}.}

\item{nearPD}{Logical.
If \code{TRUE}, attempts to coerce non–positive-definite matrices
using \code{\link[Matrix:nearPD]{Matrix::nearPD()}}.}

\item{diagnostics}{Logical.
If \code{TRUE}, returns diagnostics including McDonald's Omega and
item-level summaries.}
}
\value{
If diagnostics = FALSE, returns a correlation matrix (class: matrix).
If diagnostics = TRUE, returns a list with:
- R: correlation matrix
- Omega: per-factor Omega or adjusted Omega
- OmegaTotal: total Omega across all factors
- Diagnostics: dataframe of communalities, uniquenesses, and primary factor
}
\description{
Constructs an inter-item correlation matrix based on a user-supplied
matrix of standardised factor loadings and (optionally) a factor
correlation matrix.
The \code{makeCorrLoadings()} function does a surprisingly good job of
reproducing a target correlation matrix when all item-factor loadings
are present, as shown in the \emph{makeCorrLoadings() validation article}.
}
\examples{

# --------------------------------------------------------
# Example 1: Basic use without diagnostics
# --------------------------------------------------------

factorLoadings <- matrix(
  c(
    0.05, 0.20, 0.70,
    0.10, 0.05, 0.80,
    0.05, 0.15, 0.85,
    0.20, 0.85, 0.15,
    0.05, 0.85, 0.10,
    0.10, 0.90, 0.05,
    0.90, 0.15, 0.05,
    0.80, 0.10, 0.10
  ),
  nrow = 8, ncol = 3, byrow = TRUE
)

rownames(factorLoadings) <- paste0("Q", 1:8)
colnames(factorLoadings) <- c("Factor1", "Factor2", "Factor3")

factorCor <- matrix(
  c(
    1.0,  0.7, 0.6,
    0.7,  1.0, 0.4,
    0.6,  0.4, 1.0
  ),
  nrow = 3, byrow = TRUE
)

itemCor <- makeCorrLoadings(factorLoadings, factorCor)
round(itemCor, 3)

# --------------------------------------------------------
# Example 2: Diagnostics with factor correlations (Adjusted Omega)
# --------------------------------------------------------

result_adj <- makeCorrLoadings(
  loadings = factorLoadings,
  factorCor = factorCor,
  diagnostics = TRUE
)

# View outputs
round(result_adj$R, 3) # correlation matrix
round(result_adj$Omega, 3) # adjusted Omega
round(result_adj$OmegaTotal, 3) # total Omega
print(result_adj$Diagnostics) # communality and uniqueness per item

# --------------------------------------------------------
# Example 3: Diagnostics assuming orthogonal factors (Per-Factor Omega)
# --------------------------------------------------------

result_orth <- makeCorrLoadings(
  loadings = factorLoadings,
  diagnostics = TRUE
)

round(result_orth$Omega, 3) # per-factor Omega
round(result_orth$OmegaTotal, 3) # total Omega
print(result_orth$Diagnostics)

}
\seealso{
\itemize{
\item makeCorrLoadings() validation article:
\url{https://winzarh.github.io/LikertMakeR/articles/makeCorrLoadings_validate.html}
\item Package website: \url{https://winzarh.github.io/LikertMakeR/}
}
}
