# =============================================================================
# MODULE 1: STOCHASTICITY METRICS
# =============================================================================

#' Coefficient of Variation (MAD-based)
#'
#' Robust CV using MAD-based scale estimation.
#'
#' @param x Numeric vector
#' @return CV value, or NA_real_ if insufficient data
#' @export
#' @examples
#' cv(rnorm(50, 10, 2))
cv <- function(x) {
  x <- x[is.finite(x)]
  if (length(x) < 2) return(NA_real_)
  mu <- mean(x, na.rm = TRUE)
  if (is.na(mu) || mu == 0) return(NA_real_)
  robust_scale(x) / abs(mu)
}

#' Fano Factor (Index of Dispersion)
#'
#' Variance-to-mean ratio for positive values.
#'
#' @param x Numeric vector (positive values)
#' @return Fano factor, or NA_real_ if insufficient data
#' @export
#' @examples
#' fano_factor(rpois(100, lambda = 5))
fano_factor <- function(x) {
  x <- x[is.finite(x) & x > 0]
  if (length(x) < 2) return(NA_real_)
  mu <- mean(x, na.rm = TRUE)
  if (is.na(mu) || mu == 0) return(NA_real_)
  stats::var(x, na.rm = TRUE) / mu
}

#' Strictly Standardized Mean Difference
#'
#' Robust SSMD using MAD-based pooled variance.
#'
#' @param treatment Numeric vector of treatment values
#' @param control Numeric vector of control values
#' @return SSMD value, or NA_real_ if insufficient data
#' @export
#' @references
#' Zhang XHD (2007). A pair of new statistical parameters for quality control
#' in RNA interference high-throughput screening assays. Genomics 89(4):552-61.
#' @examples
#' ssmd(rnorm(20, 10, 2), rnorm(20, 5, 2))
ssmd <- function(treatment, control) {
  t_vals <- treatment[is.finite(treatment)]
  c_vals <- control[is.finite(control)]
  if (length(t_vals) < 2 || length(c_vals) < 2) return(NA_real_)
  mean_diff <- mean(t_vals, na.rm = TRUE) - mean(c_vals, na.rm = TRUE)
  t_scale <- robust_scale(t_vals)
  c_scale <- robust_scale(c_vals)
  pooled_var <- t_scale^2 + c_scale^2
  if (!is.finite(pooled_var) || pooled_var <= 0) return(NA_real_)
  mean_diff / sqrt(pooled_var)
}

#' Normalized Stochastic Index
#'
#' Position of treatment variance between NC (0) and PC (1).
#'
#' @param treatment_var Treatment variance
#' @param pc_var Positive control variance
#' @param nc_var Negative control variance
#' @return Value clamped to [0, 1], or NA_real_ if inputs invalid
#' @export
#' @examples
#' normalized_stochastic_index(0.5, 1.0, 0.2)
normalized_stochastic_index <- function(treatment_var, pc_var, nc_var) {
  if (is.na(pc_var) || is.na(nc_var) || !is.finite(pc_var) || !is.finite(nc_var))
    return(NA_real_)
  denom <- pc_var - nc_var
  if (abs(denom) < .Machine$double.eps) return(NA_real_)
  raw <- (treatment_var - nc_var) / denom
  pmin(1, pmax(0, raw))
}

#' Signal-to-Noise Ratio vs Control
#'
#' Mean difference divided by control MAD.
#'
#' @param treatment Numeric vector of treatment values
#' @param control Numeric vector of control values
#' @return SNR value, or NA_real_ if insufficient data
#' @export
#' @examples
#' snr_vs_control(rnorm(20, 10, 2), rnorm(20, 5, 2))
snr_vs_control <- function(treatment, control) {
  t_vals <- treatment[is.finite(treatment)]
  c_vals <- control[is.finite(control)]
  if (length(t_vals) < 2 || length(c_vals) < 2) return(NA_real_)
  mean_diff <- mean(t_vals, na.rm = TRUE) - mean(c_vals, na.rm = TRUE)
  control_scale <- robust_scale(c_vals)
  if (!is.finite(control_scale) || control_scale == 0) return(NA_real_)
  mean_diff / control_scale
}
