localdecodeHMM <- function(x, HMM, obsdist){

  # Validate observation distribution
  if(!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("observation distribution is not supported")
  }

  # Extract model parameters
  J <- length(HMM$estimate$delta)
  Pi <- HMM$estimate$Pi
  obspar <- HMM$estimate[1:(length(HMM$estimate)-2)]

  # Use stationary distribution if delta not available
  delta <- solve(t(diag(J)-Pi+1), rep(1, J))


  n <- length(x)
  fwd <- bwd <- matrix(0, J, n)
  obsprobs <- obsprobs(x, J, obsdist, obspar)

  # Forward pass with log-sum-exp for numerical stability
  alpha <- delta * obsprobs[1,]
  ll <- log(sum(alpha))  # Cumulative log-likelihood
  alpha <- alpha/sum(alpha)  # Normalize
  fwd[,1] <- log(alpha) + ll

  for(i in 2:n){
    alpha <- alpha %*% Pi * obsprobs[i,]
    ll <- ll + log(sum(alpha))
    alpha <- alpha/sum(alpha)
    fwd[,i] <- log(alpha) + ll
  }

  # Backward pass
  bwd[,n] <- rep(0, J)  # Log(1) = 0 for final states
  alpha <- rep(1/J, J)
  ll <- log(J)

  for(i in (n-1):1){
    alpha <- Pi %*% (obsprobs[i+1,] * alpha)
    bwd[,i] <- log(alpha) + ll
    sumalpha <- sum(alpha)
    alpha <- alpha / sumalpha  # Normalize
    ll <- ll + log(sumalpha)
  }

  # Compute total log-likelihood using log-sum-exp trick
  c <- max(fwd[,n])
  loglik <- c + log(sum(exp(fwd[,n]-c)))

  # Compute posterior state probabilities
  stateprobs <- matrix(0, J, n)
  for (i in 1:n){
    stateprobs[,i] <- exp(fwd[,i] + bwd[,i] - loglik)
  }

  # Local decoding: most likely state at each time point
  ld <- numeric(n)
  for(i in 1:n){
    ld[i] <- which.max(stateprobs[,i])
  }

  return(ld)
}
