% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/permutation.R
\name{perm_test}
\alias{perm_test}
\alias{print.perm_test}
\alias{summary.perm_test}
\alias{tidy.perm_test}
\alias{glance.perm_test}
\title{Permutation test for weight informativeness in survey regression}
\usage{
perm_test(
  model,
  stat = c("pred_mean", "coef_mahal"),
  B = 1000,
  coef_subset = NULL,
  block = NULL,
  normalize = TRUE,
  engine = c("C++", "R"),
  custom_fun = NULL,
  na.action = stats::na.omit
)

\method{print}{perm_test}(x, ...)

\method{summary}{perm_test}(object, ...)

\method{tidy}{perm_test}(x, ...)

\method{glance}{perm_test}(x, ...)
}
\arguments{
\item{model}{An object of class \code{svyglm} (currently supports Gaussian family best).}

\item{stat}{Statistic to use. Options:
  \itemize{
    \item \code{"pred_mean"}:
      Compares the mean predicted outcome under weighted vs. unweighted regression.
      Simple, interpretable, and directly tied to differences in fitted population means.
      Sensitive to shifts in overall prediction levels caused by informative weights.

    \item \code{"coef_mahal"}:
      Computes the Mahalanobis distance between the weighted and unweighted coefficient vectors,
      using the unweighted precision matrix (\eqn{X'X}) as the metric. Captures joint shifts in regression coefficients,
      not just mean predictions. More powerful when informativeness manifests as changes in slopes
      or multiple coefficients simultaneously.
  }}

\item{B}{Number of permutations (e.g., 1000).}

\item{coef_subset}{Optional character vector of coefficient names to include.}

\item{block}{Optional factor for blockwise permutations (e.g., strata), permute within levels.}

\item{normalize}{Logical; if TRUE (default), normalize weights to have mean 1.}

\item{engine}{\code{"C++"} for fast WLS or \code{"R"} for pure R loop.}

\item{custom_fun}{Optional function(model, X, y, wts) -> scalar statistic (overrides \code{stat}).}

\item{na.action}{Function to handle missing data.}

\item{x}{An object of class perm_test}

\item{...}{Additional arguments passed to methods}

\item{object}{An object of class perm_test}
}
\value{
An object of class \code{"perm_test"} with fields:
  \item{stat_obs}{Observed statistic with actual weights}
  \item{stat_null}{Baseline statistic under equal weights (for centering)}
  \item{perm_stats}{Vector of permutation statistics}
  \item{p.value}{Permutation p-value (two-sided, centered at baseline)}
  \item{effect}{Observed minus median of permutation stats}
  \item{stat}{Statistic name}
  \item{B}{Number of permutations}
  \item{call}{Matched call}
  \item{method}{Description string}
}
\description{
Non-parametric test that permutes survey weights (optionally within blocks)
to generate the null distribution of a chosen statistic. Supports fast closed-form
WLS (linear case) via C++ and a pure R engine.
}
\details{
This procedure implements a non‑parametric randomization test for the
informativeness of survey weights. The null hypothesis is that, conditional
on the covariates \eqn{X}, the survey weights \eqn{w} are \emph{non‑informative}
with respect to the outcome \eqn{y}. Under this null, permuting the weights
across observations should not change the distribution of any statistic that
measures the effect of weighting.

The algorithm is:
\enumerate{
  \item Fit the unweighted regression
    \deqn{\hat\beta_{U} = (X^\top X)^{-1} X^\top y}
    and the weighted regression
    \deqn{\hat\beta_{W} = (X^\top W X)^{-1} X^\top W y,}
    where \eqn{W = \mathrm{diag}(w)}.

  \item Compute the observed test statistic \eqn{T_{\mathrm{obs}}}:
    \itemize{
      \item For \code{"pred_mean"}: the difference in mean predicted outcomes
            between weighted and unweighted fits.
      \item For \code{"coef_mahal"}: the Mahalanobis distance
            \deqn{T = (\hat\beta_{W} - \hat\beta_{U})^\top
                       (X^\top X)(\hat\beta_{W} - \hat\beta_{U}),}
            using the unweighted precision matrix as the metric.
      \item For a user‑supplied \code{custom_fun}, any scalar function of
            \eqn{(X,y,w)}.
    }

  \item Generate the null distribution by permuting the weights:
    \deqn{w^{*(b)} = P_b w, \quad b=1,\ldots,B,}
    where each \eqn{P_b} is a permutation matrix. If a \code{block} factor
    is supplied, permutations are restricted within block levels.

  \item Recompute the test statistic \eqn{T^{*(b)}} for each permuted weight
    vector. The empirical distribution of \eqn{T^{*(b)}} represents the null
    distribution under non‑informative weights.

  \item The two‑sided permutation p‑value is
    \deqn{p = \frac{1 + \sum_{b=1}^B I\{|T^{*(b)} - T_0| \ge |T_{\mathrm{obs}} - T_0|\}}
                {B+1},}
    where \eqn{T_0} is the baseline statistic under equal weights.
}

Intuitively, if the weights are informative, the observed statistic will lie
in the tails of the permutation distribution, leading to a small p‑value.
If the weights are non‑informative, shuffling them destroys any spurious
association with the outcome, and the observed statistic is typical of the
permutation distribution.
}
\examples{
# Load in survey package (required) and load in example data
library(survey)
data(api, package = "survey")

# Create a survey design and fit a weighted regression model
des <- svydesign(id = ~1, strata = ~stype, weights = ~pw, data = apistrat)
fit <- svyglm(api00 ~ ell + meals, design = des)

# Run permutation diagnostic test; reports permutation statistics with p-value
results <- perm_test(fit, stat = "pred_mean", B = 1000, engine = "R")
print(results)

}
