% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_spatial_fold.R
\name{make_spatial_fold}
\alias{make_spatial_fold}
\title{Create spatially independent training and testing folds}
\usage{
make_spatial_fold(
  data = NULL,
  dependent.variable.name = NULL,
  xy.i = NULL,
  xy = NULL,
  distance.step.x = NULL,
  distance.step.y = NULL,
  training.fraction = 0.8
)
}
\arguments{
\item{data}{Data frame containing response variable and predictors. Required only for binary response variables.}

\item{dependent.variable.name}{Character string with the name of the response variable. Must be a column name in \code{data}. Required only for binary response variables.}

\item{xy.i}{Single-row data frame with columns "x" (longitude), "y" (latitude), and "id" (record identifier). Defines the focal point from which the buffer grows.}

\item{xy}{Data frame with columns "x" (longitude), "y" (latitude), and "id" (record identifier). Contains all spatial coordinates for the dataset.}

\item{distance.step.x}{Numeric value specifying the buffer growth increment along the x-axis. Default: \code{NULL} (automatically set to 1/1000th of the x-coordinate range).}

\item{distance.step.y}{Numeric value specifying the buffer growth increment along the y-axis. Default: \code{NULL} (automatically set to 1/1000th of the y-coordinate range).}

\item{training.fraction}{Numeric value between 0.1 and 0.9 specifying the fraction of records to include in the training fold. Default: \code{0.8}.}
}
\value{
List with two elements:
\itemize{
\item \code{training}: Integer vector of record IDs (from \code{xy$id}) in the training fold.
\item \code{testing}: Integer vector of record IDs (from \code{xy$id}) in the testing fold.
}
}
\description{
Generates two spatially independent data folds by growing a rectangular buffer from a focal point until a specified fraction of records falls inside. Used internally by \code{\link[=make_spatial_folds]{make_spatial_folds()}} and \code{\link[=rf_evaluate]{rf_evaluate()}} for spatial cross-validation.
}
\details{
This function creates spatially independent training and testing folds for spatial cross-validation. The algorithm works as follows:
\enumerate{
\item Starts with a small rectangular buffer centered on the focal point (\code{xy.i})
\item Grows the buffer incrementally by \code{distance.step.x} and \code{distance.step.y}
\item Continues growing until the buffer contains the desired number of records (\verb{training.fraction * total records})
\item Assigns records inside the buffer to training and records outside to testing
}

\strong{Special handling for binary response variables:}

When \code{data} and \code{dependent.variable.name} are provided and the response is binary (0/1), the function ensures that \code{training.fraction} applies to the number of presences (1s), not total records. This prevents imbalanced sampling in presence-absence models.
}
\examples{
data(plants_df, plants_xy)

# Create spatial fold centered on first coordinate
fold <- make_spatial_fold(
  xy.i = plants_xy[1, ],
  xy = plants_xy,
  training.fraction = 0.6
)

# View training and testing record IDs
fold$training
fold$testing

# Visualize the spatial split (training = red, testing = blue, center = black)
if (interactive()) {
  plot(plants_xy[c("x", "y")], type = "n", xlab = "", ylab = "")
  points(plants_xy[fold$training, c("x", "y")], col = "red4", pch = 15)
  points(plants_xy[fold$testing, c("x", "y")], col = "blue4", pch = 15)
  points(plants_xy[1, c("x", "y")], col = "black", pch = 15, cex = 2)
}

}
\seealso{
\code{\link[=make_spatial_folds]{make_spatial_folds()}}, \code{\link[=rf_evaluate]{rf_evaluate()}}, \code{\link[=is_binary]{is_binary()}}

Other preprocessing: 
\code{\link{auto_cor}()},
\code{\link{auto_vif}()},
\code{\link{case_weights}()},
\code{\link{default_distance_thresholds}()},
\code{\link{double_center_distance_matrix}()},
\code{\link{is_binary}()},
\code{\link{make_spatial_folds}()},
\code{\link{the_feature_engineer}()},
\code{\link{weights_from_distance_matrix}()}
}
\concept{preprocessing}
