% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smvr.R, R/parse.R
\name{smvr}
\alias{smvr}
\alias{parse_semver}
\title{A vector representing versions following Semantic Versioning}
\usage{
smvr(major = integer(), minor = 0L, patch = 0L, pre_release = "", build = "")

parse_semver(x)
}
\arguments{
\item{major, minor, patch}{Non-negative integers representing
the major, minor, and patch version components.
The default values for \code{minor} and \code{patch} are \code{0}.}

\item{pre_release}{Something that can be cast to a \link{pre_release_ids} vector.
This can be empty (\code{""}) meaning non pre-release (default).}

\item{build}{Optional build metadata character vector.
Should have the pattern \verb{^[a-zA-Z0-9-]+} and can contain
multiple components separated by dots (\code{.}).
This can be empty (\code{""}) meaning no build metadata (default).}

\item{x}{A character vector representing semantic versions.
Each version should follow the
\href{https://semver.org/}{Semantic Versioning Specification}.
Partial matches are not allowed (e.g., \code{"1.0"} is not valid).}
}
\value{
A \link{smvr} class vector.
}
\description{
The \code{smvr} class represents versions that follow the
\href{https://semver.org/}{Semantic Versioning Specification (SemVer)}.
A version number contains three components, \code{MAJOR.MINOR.PATCH},
and optional pre-release and build metadata labels.

This is similar to the base R's \link{numeric_version} class, but always has
three components (major, minor, patch) and supports pre-release
and build metadata labels. And, unlike \link{numeric_version},
SemVer uses dots (\code{.}) as separators and does not allow hyphens (\code{-})
except to indicate the start of a pre-release label.

There are two functions to create \link{smvr} objects:
\itemize{
\item \code{\link[=smvr]{smvr()}} is a constructor from each component.
Each component must have the same length or length 1 (will be recycled).
\item \code{\link[=parse_semver]{parse_semver()}} parses a character vector.
}
}
\details{
Build metadata is not used for ordering, but the \code{==} and \code{!=} operators
check it and exactly same build metadata is required for equality.
The other operators (\code{<}, \code{<=}, \code{>}, \code{>=}) ignore build metadata.
}
\examples{
# SemVer versions from components
smvr(4, 1:5)

# Parse SemVer versions from character
v <- parse_semver(c(
  "1.0.0",
  "1.0.0-alpha",
  "1.0.0-beta",
  "1.0.0-rc.2",
  "1.0.0-rc.10",
  NA
))
v

# Sorting
vctrs::vec_sort(v)

# Can be compared with string notation
v[v >= "1.0.0-rc.2" & !is.na(v)]

# Partial version components are treated as NA
suppressWarnings(parse_semver("1.5"))

# The numeric_version class supports versions with
# less than 3 components, and can be cast to smvr.
numeric_version("1.5") |>
  vctrs::vec_cast(smvr())

# Be careful with hyphens in numeric_version and SemVer.
# The following examples yield opposite results.
numeric_version("1.0.0-1") > "1.0.0" # 1.0.0-1 is the same as 1.0.0.1
parse_semver("1.0.0-1") > "1.0.0"    # 1.0.0-1 is a pre-release version
}
\seealso{
\itemize{
\item \code{\link[=as_smvr]{as_smvr()}} to convert other classes to \link{smvr}.
\item \link{extract-component} functions to extract components of a \link{smvr} object.
(Operations opposite to \code{\link[=smvr]{smvr()}}).
\item \link{update-version} functions to update components of a \link{smvr} object.
}
}
