% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-fixed.R
\name{SimSSMFixed}
\alias{SimSSMFixed}
\title{Simulate Data from a State Space Model
(Fixed Parameters)}
\usage{
SimSSMFixed(
  n,
  time,
  delta_t = 1,
  mu0,
  sigma0_l,
  alpha,
  beta,
  psi_l,
  nu,
  lambda,
  theta_l,
  type = 0,
  x = NULL,
  gamma = NULL,
  kappa = NULL
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{time}{Positive integer.
Number of time points.}

\item{delta_t}{Numeric.
Time interval.
The default value is \code{1.0}
with an option to use a numeric value
for the discretized state space model
parameterization of the
linear stochastic differential equation model.}

\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(sigma0))})
of the covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{alpha}{Numeric vector.
Vector of constant values for the dynamic model
(\eqn{\boldsymbol{\alpha}}).}

\item{beta}{Numeric matrix.
Transition matrix relating the values of the latent variables
at the previous to the current time point
(\eqn{\boldsymbol{\beta}}).}

\item{psi_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(psi))})
of the covariance matrix
of the process noise
(\eqn{\boldsymbol{\Psi}}).}

\item{nu}{Numeric vector.
Vector of intercept values for the measurement model
(\eqn{\boldsymbol{\nu}}).}

\item{lambda}{Numeric matrix.
Factor loading matrix linking the latent variables
to the observed variables
(\eqn{\boldsymbol{\Lambda}}).}

\item{theta_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(theta))})
of the covariance matrix
of the measurement error
(\eqn{\boldsymbol{\Theta}}).}

\item{type}{Integer.
State space model type.
See Details for more information.}

\item{x}{List.
Each element of the list is a matrix of covariates
for each individual \code{i} in \code{n}.
The number of columns in each matrix
should be equal to \code{time}.}

\item{gamma}{Numeric matrix.
Matrix linking the covariates to the latent variables
at current time point
(\eqn{\boldsymbol{\Gamma}}).}

\item{kappa}{Numeric matrix.
Matrix linking the covariates to the observed variables
at current time point
(\eqn{\boldsymbol{\kappa}}).}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers with length \code{l},
where \code{l} is the value of the function argument \code{time}.
\item \code{time}: A vector time points of length \code{l}.
\item \code{y}: A \code{l} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{l} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{l} by \code{j} matrix of values for the covariates
(when covariates are included).
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data using a state space model.
It assumes that the parameters remain constant
across individuals and over time.
}
\details{
\subsection{Type 0}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\mathbf{y}_{i, t}},
\eqn{\boldsymbol{\eta}_{i, t}},
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
are random variables
and
\eqn{\boldsymbol{\nu}},
\eqn{\boldsymbol{\Lambda}},
and
\eqn{\boldsymbol{\Theta}}
are model parameters.
\eqn{\mathbf{y}_{i, t}}
represents a vector of observed random variables,
\eqn{\boldsymbol{\eta}_{i, t}}
a vector of latent random variables,
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
a vector of random measurement errors,
at time \eqn{t} and individual \eqn{i}.
\eqn{\boldsymbol{\nu}}
denotes a vector of intercepts,
\eqn{\boldsymbol{\Lambda}}
a matrix of factor loadings,
and
\eqn{\boldsymbol{\Theta}}
the covariance matrix of
\eqn{\boldsymbol{\varepsilon}}.

An alternative representation of the measurement error
is given by
\deqn{
    \boldsymbol{\varepsilon}_{i, t}
    =
    \boldsymbol{\Theta}^{\frac{1}{2}}
    \mathbf{z}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \mathbf{z}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \mathbf{I}
    \right)
  }
where
\eqn{\mathbf{z}_{i, t}} is a vector of
independent standard normal random variables and
\eqn{
    \left( \boldsymbol{\Theta}^{\frac{1}{2}} \right)
    \left( \boldsymbol{\Theta}^{\frac{1}{2}} \right)^{\prime}
    =
    \boldsymbol{\Theta} .
  }

The dynamic structure is given by
\deqn{
    \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\alpha}
    +
    \boldsymbol{\beta}
    \boldsymbol{\eta}_{i, t - 1}
    +
    \boldsymbol{\zeta}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\zeta}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Psi}
    \right)
  }
where
\eqn{\boldsymbol{\eta}_{i, t}},
\eqn{\boldsymbol{\eta}_{i, t - 1}},
and
\eqn{\boldsymbol{\zeta}_{i, t}}
are random variables,
and
\eqn{\boldsymbol{\alpha}},
\eqn{\boldsymbol{\beta}},
and
\eqn{\boldsymbol{\Psi}}
are model parameters.
Here,
\eqn{\boldsymbol{\eta}_{i, t}}
is a vector of latent variables
at time \eqn{t} and individual \eqn{i},
\eqn{\boldsymbol{\eta}_{i, t - 1}}
represents a vector of latent variables
at time \eqn{t - 1} and individual \eqn{i},
and
\eqn{\boldsymbol{\zeta}_{i, t}}
represents a vector of dynamic noise
at time \eqn{t} and individual \eqn{i}.
\eqn{\boldsymbol{\alpha}}
denotes a vector of intercepts,
\eqn{\boldsymbol{\beta}}
a matrix of autoregression
and cross regression coefficients,
and
\eqn{\boldsymbol{\Psi}}
the covariance matrix of
\eqn{\boldsymbol{\zeta}_{i, t}}.

An alternative representation of the dynamic noise
is given by
\deqn{
    \boldsymbol{\zeta}_{i, t}
    =
    \boldsymbol{\Psi}^{\frac{1}{2}}
    \mathbf{z}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \mathbf{z}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \mathbf{I}
    \right)
  }
where
\eqn{
    \left( \boldsymbol{\Psi}^{\frac{1}{2}} \right)
    \left( \boldsymbol{\Psi}^{\frac{1}{2}} \right)^{\prime}
    =
    \boldsymbol{\Psi} .
  }
}

\subsection{Type 1}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right) .
  }

The dynamic structure is given by
\deqn{
    \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\alpha}
    +
    \boldsymbol{\beta}
    \boldsymbol{\eta}_{i, t - 1}
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\zeta}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\zeta}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Psi}
    \right)
  }
where
\eqn{\mathbf{x}_{i, t}} represents a vector of covariates
at time \eqn{t} and individual \eqn{i},
and \eqn{\boldsymbol{\Gamma}} the coefficient matrix
linking the covariates to the latent variables.
}

\subsection{Type 2}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\kappa}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\boldsymbol{\kappa}} represents the coefficient matrix
linking the covariates to the observed variables.

The dynamic structure is given by
\deqn{
    \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\alpha}
    +
    \boldsymbol{\beta}
    \boldsymbol{\eta}_{i, t - 1}
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\zeta}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\zeta}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Psi}
    \right) .
  }
}
}
\examples{
# prepare parameters
set.seed(42)
## number of individuals
n <- 5
## time points
time <- 50
## dynamic structure
p <- 3
mu0 <- rep(x = 0, times = p)
sigma0 <- 0.001 * diag(p)
sigma0_l <- t(chol(sigma0))
alpha <- rep(x = 0, times = p)
beta <- 0.50 * diag(p)
psi <- 0.001 * diag(p)
psi_l <- t(chol(psi))
## measurement model
k <- 3
nu <- rep(x = 0, times = k)
lambda <- diag(k)
theta <- 0.001 * diag(k)
theta_l <- t(chol(theta))
## covariates
j <- 2
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    matrix(
      data = stats::rnorm(n = time * j),
      nrow = j,
      ncol = time
    )
  }
)
gamma <- diag(x = 0.10, nrow = p, ncol = j)
kappa <- diag(x = 0.10, nrow = k, ncol = j)

# Type 0
ssm <- SimSSMFixed(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  alpha = alpha,
  beta = beta,
  psi_l = psi_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 0
)

plot(ssm)

# Type 1
ssm <- SimSSMFixed(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  alpha = alpha,
  beta = beta,
  psi_l = psi_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 1,
  x = x,
  gamma = gamma
)

plot(ssm)

# Type 2
ssm <- SimSSMFixed(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  alpha = alpha,
  beta = beta,
  psi_l = psi_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 2,
  x = x,
  gamma = gamma,
  kappa = kappa
)

plot(ssm)

}
\references{
Chow, S.-M., Ho, M. R., Hamaker, E. L., & Dolan, C. V. (2010).
Equivalence and differences between structural equation modeling
and state-space modeling techniques.
\emph{Structural Equation Modeling: A Multidisciplinary Journal},
17(2), 303–332.
\doi{10.1080/10705511003661553}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{LinSDE2SSM}()},
\code{\link{LinSDECovEta}()},
\code{\link{LinSDECovY}()},
\code{\link{LinSDEMeanEta}()},
\code{\link{LinSDEMeanY}()},
\code{\link{ProjectToHurwitz}()},
\code{\link{ProjectToStability}()},
\code{\link{SSMCovEta}()},
\code{\link{SSMCovY}()},
\code{\link{SSMMeanEta}()},
\code{\link{SSMMeanY}()},
\code{\link{SimAlphaN}()},
\code{\link{SimBetaN}()},
\code{\link{SimBetaN2}()},
\code{\link{SimBetaNCovariate}()},
\code{\link{SimCovDiagN}()},
\code{\link{SimCovN}()},
\code{\link{SimIotaN}()},
\code{\link{SimNuN}()},
\code{\link{SimPhiN}()},
\code{\link{SimPhiN2}()},
\code{\link{SimPhiNCovariate}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinSDEFixed}()},
\code{\link{SimSSMLinSDEIVary}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{SpectralRadius}()},
\code{\link{TestPhi}()},
\code{\link{TestPhiHurwitz}()},
\code{\link{TestStability}()},
\code{\link{TestStationarity}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{sim}
\keyword{simStateSpace}
\keyword{ssm}
