% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spec_interp_area_targets.R
\name{spec_interp_area_targets}
\alias{spec_interp_area_targets}
\title{Specify targets based on interpolating area-based thresholds}
\usage{
spec_interp_area_targets(
  rare_area_threshold,
  rare_relative_target,
  rare_area_target,
  rare_method,
  common_area_threshold,
  common_relative_target,
  common_area_target,
  common_method,
  cap_area_target,
  interp_method,
  area_units
)
}
\arguments{
\item{rare_area_threshold}{\code{numeric} value indicating the threshold
area for identifying rare features.
This value must be expressed in the same units as the feature data.
In particular, features with a total spatial extent
smaller than this value will be considered rare during the target setting
calculations.}

\item{rare_relative_target}{\code{numeric} value indicating the
relative target for rare features.
Note that this value must be a proportion between 0 and 1.
For example, a value of 0.1 corresponds to 10\%.}

\item{rare_area_target}{\code{numeric} value denoting the
area-based target for rare features.
This value must be expressed in the same units as \code{area_units}.
To avoid setting an area-based target for rare features,
a missing (\code{NA}) value can be specified.}

\item{rare_method}{\code{character} value indicating how the target for rare
features should be calculated. Available options include \code{"min"} and \code{"max"}.
For example, a value of \code{"max"} means that the target for a rare features
is calculated as the maximum based on \code{rare_relative_target} and
\code{rare_area_threshold}. Note that \code{rare_method} will have no effect on
the target calculations if \code{rare_area_target} is a missing (\code{NA}) value.}

\item{common_area_threshold}{\code{numeric} value indicating the
threshold area for identifying common features.
This value must be expressed in the same units as \code{area_units}.
In particular, features with a total spatial extent
greater than this value will be considered common during the target setting
calculations.}

\item{common_relative_target}{\code{numeric} value denoting the
relative target for common features.
Note that this value must be a proportion between 0 and 1.
For example, a value of 0.1 corresponds to 10\%.}

\item{common_area_target}{\code{numeric} value denoting the
area-based target for common features.
This value must be expressed in the same units as \code{area_units}.
To avoid setting an area-based target for common features,
a missing (\code{NA}) value can be specified.}

\item{common_method}{\code{character} value indicating how the target for common
features should be calculated. Available options include \code{"min"} and \code{"max"}.
For example, a value of \code{"max"} means that the target for a common feature
is calculated as the maximum based on \code{common_relative_target} and
\code{common_area_threshold}. Note that \code{common_method} will have
no effect on the target calculations if \code{common_area_target} is a
missing (\code{NA}) value.}

\item{cap_area_target}{\code{numeric} value denoting the area-based target
cap.
This value must be expressed in the same units as \code{area_units}.
In particular, all targets are clamped to this value during target setting
calculations.
To avoid setting a target cap,
a missing (\code{NA}) value can be specified.}

\item{interp_method}{\code{character} value denoting the interpolation method.
Available options include \code{"linear"} for linear interpolation and
\code{"log10"} for log-linear interpolation.}

\item{area_units}{\code{character} value denoting the unit of measurement
for the area-based arguments (e.g., \verb{"km^2", }"ha"\verb{, }"acres"`).}
}
\value{
An object (\code{\linkS4class{TargetMethod}}) for specifying targets that
can be used with \code{\link[=add_auto_targets]{add_auto_targets()}} and \code{\link[=add_group_targets]{add_group_targets()}}
to add targets to a \code{\link[=problem]{problem()}}.
}
\description{
Specify targets by interpolating them between area-based thresholds.
Briefly, this method involves
(i) setting target thresholds for rare features to a particular percentage
threshold, (ii) setting target thresholds for common features
to a particular percentage threshold, and (iii) interpolating
target thresholds for features with spatial distributions that
range between the those for the rare and common features.
Additionally, features can (optionally) have their targets capped at a
particular threshold.
This method is especially useful for setting targets based on
interpolation procedures when features have data expressed as an area-based
unit of measurement (e.g., \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).
Note that this function is designed to be used with \code{\link[=add_auto_targets]{add_auto_targets()}}
and \code{\link[=add_group_targets]{add_group_targets()}}.
}
\details{
This method has been applied to set target thresholds at global and national
scales (e.g., Butchart \emph{et al.} 2015;
Rodrigues \emph{et al.} 2004; Polak \emph{et al.} 2015).
It is based on the rationale that species with a smaller geographic
distribution are at a greater risk of extinction, and so require
a larger percentage of their geographic distribution to be represented
by a prioritization (Rodrigues \emph{et al.} 2004).
When using this method in a planning exercise, it is important to ensure
that the threshold parameters reflect the stakeholder objectives.
Additionally, the threshold parameters may need to set according to
the spatial extent of the planning region.
}
\section{Mathematical formulation}{

This method provides a flexible approach for setting target thresholds based
on an interpolation procedure and the spatial extent of the features.
To express this mathematically, we will define the following terminology.
Let \eqn{f} denote the total spatial extent of a feature (e.g., geographic
range size),
\eqn{a} the threshold for identifying rare features
(per \code{rare_area_threshold} and \code{area_units}),
\eqn{b} the relative targets for rare features
(per \code{rare_relative_target}),
\eqn{c} the area-based targets for rare features
(per \code{rare_area_target} and \code{area_units}),
\eqn{d()} the function for calculating targets for rare features
as a maximum or minimum value (per \code{rare_method}),
\eqn{e} the threshold for identifying common features
(per \code{common_area_threshold} and \code{area_units}),
\eqn{g} the relative targets for common features
(per \code{common_relative_target}),
\eqn{h} the area-based targets for common features
(per \code{common_area_target} and \code{area_units}),
\eqn{i()} the method for calculating targets for common features
as a maximum or minimum value  (per \code{common_method}), and
\eqn{j} the target cap (per \code{cap_area_target} and \code{area_units}), and
\eqn{k()} the interpolation method for features with a spatial distribution
that is larger than a rare features and smaller than a common feature
(per \code{interp_method}).
In particular, \eqn{k()} is either a linear or log-linear interpolation
procedure based on the thresholds for identifying rare and common features
as well as the relative targets for rare and common features.
Given this terminology, the target threshold (\eqn{t}) for the feature
is calculated as follows.
\itemize{
\item If \eqn{f < a}, then \eqn{
t = min(d(c, b \times f), j)}{
t = min(d(c, b * f), j)
}.
\item If \eqn{f > e}, then \eqn{
t = min(i(h, g \times f), j)}{.
t = min(i(h, g * f), j)
}.
\item If \eqn{a \leq f \leq e}{a <= f <= e}, then
\eqn{t = min(k(f, a, b, e, g), j)}.
}
}

\section{Data calculations}{

This function involves calculating targets based on the spatial extent
of the features in \code{x}.
Although it can be readily applied to \code{\link[=problem]{problem()}} objects that
have the feature data provided as a \code{\link[terra:rast]{terra::rast()}} object,
you will need to specify the spatial units for the features
when initializing the \code{\link[=problem]{problem()}} objects if the feature data
are provided in a different format. In particular, if the feature
data are provided as a \code{data.frame} or \code{character} vector,
then you will need to specify an argument to \code{feature_units} when
using the \code{\link[=problem]{problem()}} function.
See the Examples section of the documentation for \code{\link[=add_auto_targets]{add_auto_targets()}}
for a demonstration of specifying the spatial units for features.
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_complex_pu_raster <- get_sim_complex_pu_raster()
sim_complex_features <- get_sim_complex_features()

# create problem with interpolated targets.
# here, targets will be set as 100\% for features smaller than 1000 km^2
# in size, 10\% for features greater than 250,000 km^2 in size,
# log-linearly interpolated for features with an intermediate range size,
# and capped at 1,000,000 km^2
p1 <-
  problem(sim_complex_pu_raster, sim_complex_features) \%>\%
  add_min_set_objective() \%>\%
  add_auto_targets(
    method = spec_interp_area_targets(
     rare_area_threshold = 1000,
     rare_relative_target = 1,
     rare_area_target = NA,            # not used
     rare_method = "max",              # not used
     common_area_threshold = 250000,
     common_relative_target = 0.1,
     common_area_target = NA,          # not used
     common_method = "max",            # not used
     cap_area_target = 1000000,
     interp_method = "log10",
     area_units = "km^2"
    )
  ) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE)
}
}
\references{
Butchart SHM, Clarke M, Smith RJ, Sykes RE, Scharlemann JPW, Harfoot M,
Buchanan GM, Angulo A, Balmford A, Bertzky B, Brooks TM, Carpenter KE,
Comeros‐Raynal MT, Cornell J, Ficetola GF, Fishpool LDC, Fuller RA,
Geldmann J, Harwell H, Hilton‐Taylor C, Hoffmann M, Joolia A, Joppa L,
Kingston N, May I, Milam A, Polidoro B, Ralph G, Richman N, Rondinini C,
Segan DB, Skolnik B, Spalding MD, Stuart SN, Symes A, Taylor J, Visconti P,
Watson JEM, Wood L, Burgess ND (2015) Shortfalls and solutions for meeting
national and global conservation area targets. \emph{Conservation Letters},
8: 329--337.

Polak T, Watson JEM, Fuller RA, Joseph LN, Martin TG, Possingham HP,
Venter O, Carwardine J (2015) Efficient expansion of global protected areas
requires simultaneous planning for species and ecosystems.
\emph{Royal Society Open Science}, 2: 150107.

Rodrigues ASL, Akçakaya HR, Andelman SJ, Bakarr MI, Boitani L, Brooks TM,
Chanson JS, Fishpool LDC, Da Fonseca GAB, Gaston KJ, Hoffmann M, Marquet PA,
Pilgrim JD, Pressey RL, Schipper J, Sechrest W, Stuart SN, Underhill LG,
Waller RW, Watts MEJ, Yan X (2004)
Global gap analysis: priority regions for expanding the global
protected-area network. \emph{BioScience}, 54: 1092--1100.
}
\seealso{
Other target setting methods:
\code{\link{spec_absolute_targets}()},
\code{\link{spec_area_targets}()},
\code{\link{spec_duran_targets}()},
\code{\link{spec_interp_absolute_targets}()},
\code{\link{spec_jung_targets}()},
\code{\link{spec_max_targets}()},
\code{\link{spec_min_targets}()},
\code{\link{spec_polak_targets}()},
\code{\link{spec_pop_size_targets}()},
\code{\link{spec_relative_targets}()},
\code{\link{spec_rl_ecosystem_targets}()},
\code{\link{spec_rl_species_targets}()},
\code{\link{spec_rodrigues_targets}()},
\code{\link{spec_rule_targets}()},
\code{\link{spec_ward_targets}()},
\code{\link{spec_watson_targets}()},
\code{\link{spec_wilson_targets}()}
}
\concept{methods}
