% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidy_interpolation_cross_validation.R,
%   R/tidy_interpolator.R
\name{interpolation_cross_validation}
\alias{interpolation_cross_validation}
\alias{interpolator_calibration}
\title{Calibration and validation of interpolation procedures}
\usage{
interpolation_cross_validation(
  interpolator,
  stations = NULL,
  verbose = getOption("meteoland_verbosity", TRUE)
)

interpolator_calibration(
  interpolator,
  stations = NULL,
  update_interpolation_params = FALSE,
  variable = "MinTemperature",
  N_seq = seq(10, 30, by = 5),
  alpha_seq = seq(0.25, 10, by = 0.25),
  verbose = getOption("meteoland_verbosity", TRUE)
)
}
\arguments{
\item{interpolator}{A meteoland interpolator object, as created by
\code{\link{create_meteo_interpolator}}}

\item{stations}{A vector with the stations (numeric for station indexes or
character for stations id) to be used to calculate \code{"MAE"}. All
stations with data are included in the training set but predictive
\code{"MAE"} are calculated for the stations subset indicated in
\code{stations} param only. If \code{NULL} all stations are used in the
predictive \code{"MAE"} calculation.}

\item{verbose}{Logical indicating if the function must show messages and info.
Default value checks \code{"meteoland_verbosity"} option and if not set, defaults
to TRUE. It can be turned off for the function with FALSE, or session wide with
\code{options(meteoland_verbosity = FALSE)}}

\item{update_interpolation_params}{Logical indicating if the interpolator
object must be updated with the calculated parameters. Default to FALSE}

\item{variable}{A string indicating the meteorological variable for which
interpolation parameters \code{"N"} and \code{"alpha"} will be calibrated.
Accepted values are:
\itemize{
\item{\code{MinTemperature} (kernel for minimum temperature)}
\item{\code{MaxTemperature} (kernel for maximum temperature)}
\item{\code{DewTemperature} (kernel for dew-temperature (i.e. relative humidity))}
\item{\code{Precipitation} (to calibrate the same
kernel for both precipitation events and regression of precipitation amounts;
not recommended)}
\item{\code{PrecipitationAmount} (kernel for regression of precipitation amounts)}
\item{\code{PrecipitationEvent} (kernel for precipitation events)}
}}

\item{N_seq}{Numeric vector with \code{"N"} values to be tested}

\item{alpha_seq}{Numeric vector with \code{"alpha"}}
}
\value{
\code{interpolation_cross_validation} returns a list with the
following items
\itemize{
\item{errors: Data frame with each combination of station and date with
observed variables, predicated variables and the total error
(predicted - observed) calculated for each variable}
\item{station_stats: Data frame with error and bias statistics aggregated by
station}
\item{dates_stats: Data frame with error and bias statistics aggregated by
date}
\item{r2: correlation indexes between observed and predicted values for each
meteorological variable}
}

If \code{update_interpolation_params} is FALSE (default),
\code{interpolator_calibration} returns a list with the following items
\itemize{ \item{MAE: A numeric matrix with the mean absolute error values,
averaged across stations, for each combination of parameters \code{"N"} and
\code{"alpha"}} \item{minMAE: Minimum MAE value} \item{N: Value of parameter
\code{"N"} corresponding to the minimum MAE} \item{alpha: Value of parameter
\code{"alpha"} corresponding the the minimum MAE} \item{observed: matrix with
observed values (meteorological measured values)} \item{predicted: matrix with
interpolated values for the optimum parameter combination} } If
\code{update_interpolation_params} is FALSE, \code{interpolator_calibration}
returns the interpolator provided with the parameters updated
}
\description{
Calibration and validation of interpolation procedures
}
\details{
Function \code{interpolator_calibration} determines optimal interpolation
parameters \code{"N"} and \code{"alpha"} for a given meteorological
variable. Optimization is done by minimizing mean absolute error ("MAE")
(Thornton \emph{et al.} 1997). Function
\code{interpolation_cross_validation} calculates average mean absolute
errors ("MAE") for the prediction period of the interpolator object. In both
calibration and cross validation procedures, predictions for each
meteorological station are made using a \emph{leave-one-out} procedure (i.e.
after excluding the station from the predictive set).
}
\section{Functions}{
\itemize{
\item \code{interpolation_cross_validation()}: 

}}
\examples{

\donttest{
# example interpolator
data("meteoland_interpolator_example")

# As the cross validation for all stations can be time consuming, we are
# gonna use only for the first 5 stations of the 198
cv <- interpolation_cross_validation(meteoland_interpolator_example, stations = 1:5)

# Inspect the results
cv$errors
cv$station_stats
cv$dates_stats
cv$r2
}


\donttest{
# example interpolator
data("meteoland_interpolator_example")

# As the calibration for all stations can be time consuming, we are gonna
# interpolate only for the first 5 stations of the 198 and only a handful
# of parameter combinations
calibration <- interpolator_calibration(
  meteoland_interpolator_example,
  stations = 1:5,
  variable = "MaxTemperature",
  N_seq = seq(10, 20, by = 5),
  alpha_seq = seq(8, 9, by = 0.25)
)

# we can update the interpolator params directly:
updated_interpolator <- interpolator_calibration(
  meteoland_interpolator_example,
  stations = 1:5,
  update_interpolation_params = TRUE,
  variable = "MaxTemperature",
  N_seq = seq(10, 20, by = 5),
  alpha_seq = seq(8, 9, by = 0.25)
)


# check the new interpolator have the parameters updated
get_interpolation_params(updated_interpolator)$N_MaxTemperature
get_interpolation_params(updated_interpolator)$alpha_MaxTemperature
}

}
\references{
Thornton, P.E., Running, S.W., 1999. An improved algorithm for
estimating incident daily solar radiation from measurements of temperature,
humidity, and precipitation. Agric. For. Meteorol. 93, 211–228.
doi:10.1016/S0168-1923(98)00126-9.

De Caceres M, Martin-StPaul N, Turco M, Cabon A, Granda V (2018) Estimating
daily meteorological data and downscaling climate models over landscapes.
Environmental Modelling and Software 108: 186-196.
}
\author{
Miquel De \enc{Cáceres}{Caceres} Ainsa, EMF-CREAF

Victor Granda \enc{García}{Garcia}, EMF-CREAF
}
