#' Local mark correlation functions for homogeneous point patterns with function-valued marks.
#'
#' Local mark correlation functions for homogeneous point patterns with function-valued marks.
#'
#'
#' @usage lfmcorr(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart",
#'  "isham", "stoyancov", "schlather"),
#' r = NULL,
#' method = c("density","loess"),
#' normalise = TRUE,
#' f = NULL,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class ppp or lpp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \eqn{r} at which the mark correlation function should be evaluated.
#' @param method Type of smoothing, either \code{density} or \code{loess}. See details.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param tol Tolerance used in the calculation of the conditional mean of the marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' This function computes local mark correlation functions for a homogeneous point pattern with a function-valued mark. See the details of test functions used in \code{\link[markstat]{fmcorr}}. Technical details are given in Eckardt and Moradi (2025).
#' @examples
#'  library(spatstat.random)
#'  library(spatstat.geom)
#'  X <- rpoispp(100)
#'  marks(X) <- data.frame(
#'  t1 = runif(npoints(X),1,10),
#'  t2 = runif(npoints(X),1,10),
#'  t3 = runif(npoints(X),1,10),
#'  t4 = runif(npoints(X),1,10),
#'  t5 = runif(npoints(X),1,10))
#'  lfmcorr(X,  ftype = "stoyan", method = "density")

#' @return a data.frame which gives the estimated overall local mark correlation function and the distance vector \eqn{r} at which the local mark correlation function is estimated. The outputs of the local mark correlation functions for each time point are stored as an attribute, which can be extracted as \code{attr(., "ests.time")}.
#' The outputs of the local mark correlation functions for each data point are stored as an attribute, which can be extracted as \code{attr(., "ests.points")}.
#' @references 
#' Eckardt, M., Mateu, J., & Moradi, M. (2024). Function‐Valued Marked Spatial Point Processes on Linear Networks: Application to Urban Cycling Profiles. Stat, 13(4), e70013.
#' 
#' Eckardt, M., & Moradi, M. (2025). Local indicators of mark association for marked spatial point processes.
#' 
#' @seealso \code{\link[markstat]{mcorr.ppp}}, \code{\link[markstat]{mcorr.lpp}}, \code{\link[markstat]{fmcorr}}.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt



#' @import spatstat.univar
#' @import spatstat.linnet
#' @import spatstat.geom
#' @import spatstat.explore
#' @import spatstat.utils
#' @import stats
#' @export

lfmcorr <- function(X,
                    ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                    r = NULL,
                    method = c("density","loess"),
                    normalise = TRUE,
                    f = NULL,
                    tol = 0.01,
                    ...){

  if (all(class(X) != "lpp" & class(X) != "ppp")) stop("object X should be of class lpp or ppp.")

  if(all(class(marks(X)) != "data.frame" & class(marks(X)) != "hyperframe")) stop("object X should have a funtion-valued mark as a data.frame whose columns represent marks at time points.")

  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")

  if (missing(method)) stop("smoothing method should be chosen.")

  n <- npoints(X)
  d <- pairdist(X)

  if(is.null(r)){

    if(any(class(X)=="ppp")){

      W <- X$window
      rmaxdefault <- rmax.rule("K", W, n/area(W))
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else if(any(class(X)=="lpp")){

      L <- X$domain
      rmaxdefault <- 0.98 * boundingradius(L)
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      W <- Window(L)
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else {
      stop("object X should be of class lpp or ppp.")
    }
  }

  rmax <- max(r)

  m <- as.data.frame(marks(X))

  nf <- dim(m)[1]
  f.len <- dim(m)[2]

  if(any(class(X)=="ppp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- lmcorr.ppp(X, normalise = normalise, r = r, f = f, ftype = ftype, method = method, tol = tol,...)
    }

    r <- out[[1]][,"r"]
    
    col_names <- colnames(out[[1]])
    
    out.points <- lapply(X=1:length(col_names), function(i) {
      rr <- do.call(cbind, lapply(out, function(df) df[,col_names[i]]))
      colnames(rr) <- paste("p", rep(col_names[i], length(out))," ", colnames(m), sep = "")
      return(rr)
    })
    
    out.points.overall <- lapply(X=1:length(col_names), function(i){
      apply(out.points[[i]], 1, mean)
    })
    
    out.points.overall <- do.call(cbind, out.points.overall)
    
    colnames(out.points.overall) <- colnames(out[[1]])
    

  }else if(any(class(X)=="lpp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- lmcorr.lpp(X, normalise = normalise, r = r, f = f, ftype = ftype, method = method, tol = tol, ...)
    }

    r <- out[[1]][,"r"]
    
    col_names <- colnames(out[[1]])
    
    out.points <- lapply(X=1:length(col_names), function(i) {
      rr <- do.call(cbind, lapply(out, function(df) df[,col_names[i]]))
      colnames(rr) <- paste("p", rep(col_names[i], length(out))," ", colnames(m), sep = "")
      return(rr)
    })
    
    out.points.overall <- lapply(X=1:length(col_names), function(i){
      apply(out.points[[i]], 1, mean)
    })
    
    out.points.overall <- do.call(cbind, out.points.overall)
    
    colnames(out.points.overall) <- colnames(out[[1]])

  }else {
    stop("object X should be of class lpp or ppp.")
  }

  names(out) <- colnames(m)
  names(out.points) <- colnames(out[[1]])
  
 
  if(ncol(out.points.overall) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out.points.overall) <- "mc"
  attr(out.points.overall, "mtype") <- "function-valued"
  attr(out.points.overall, "ests.points") <- out.points
  attr(out.points.overall, "ests.time") <- out
  
  attr(out.points.overall, "type") <- type
  attr(out.points.overall, "ests") <- out
  attr(out.points.overall , "ftype") <- ftype
  attr(out.points.overall , "method") <- method
  attr(out.points.overall , "normalise") <- normalise
  

  return(out.points.overall)


}
