/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE_STATIC(H5HF_t);

herr_t
H5HF__op_read(const void *obj, size_t obj_len, void *op_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(op_data, obj, obj_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__op_write(const void *obj, size_t obj_len, void *op_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5_WARN_CAST_AWAY_CONST_OFF
    H5MM_memcpy((void *)obj, op_data, obj_len);
    H5_WARN_CAST_AWAY_CONST_ON

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

H5HF_t *
H5HF_create(H5F_t *f, const H5HF_create_t *cparam)
{
    H5HF_t     *fh  = NULL;       
    H5HF_hdr_t *hdr = NULL;       
    haddr_t     fh_addr;          
    H5HF_t     *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(cparam);

    
    if (HADDR_UNDEF == (fh_addr = H5HF__hdr_create(f, cparam)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't create fractal heap header");

    
    if (NULL == (fh = H5FL_MALLOC(H5HF_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed for fractal heap info");

    
    if (NULL == (hdr = H5HF__hdr_protect(f, fh_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect fractal heap header");

    
    fh->hdr = hdr;
    if (H5HF__hdr_incr(fh->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL, "can't increment reference count on shared heap header");

    
    if (H5HF__hdr_fuse_incr(fh->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared heap header");

    
    fh->f = f;

    
    ret_value = fh;

done:
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, fh_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, NULL, "unable to release fractal heap header");
    if (!ret_value && fh)
        if (H5HF_close(fh) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTCLOSEOBJ, NULL, "unable to close fractal heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_t *
H5HF_open(H5F_t *f, haddr_t fh_addr)
{
    H5HF_t     *fh        = NULL; 
    H5HF_hdr_t *hdr       = NULL; 
    H5HF_t     *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(H5_addr_defined(fh_addr));

    
    if (NULL == (hdr = H5HF__hdr_protect(f, fh_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect fractal heap header");

    
    if (hdr->pending_delete)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, NULL, "can't open fractal heap pending deletion");

    
    if (NULL == (fh = H5FL_MALLOC(H5HF_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed for fractal heap info");

    
    fh->hdr = hdr;
    if (H5HF__hdr_incr(fh->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL, "can't increment reference count on shared heap header");

    
    if (H5HF__hdr_fuse_incr(fh->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared heap header");

    
    fh->f = f;

    
    ret_value = fh;

done:
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, fh_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, NULL, "unable to release fractal heap header");
    if (!ret_value && fh)
        if (H5HF_close(fh) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTCLOSEOBJ, NULL, "unable to close fractal heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_get_id_len(H5HF_t *fh, size_t *id_len_p)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(id_len_p);

    
    *id_len_p = fh->hdr->id_len;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_get_heap_addr(const H5HF_t *fh, haddr_t *heap_addr_p)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(heap_addr_p);

    
    *heap_addr_p = fh->hdr->heap_addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_insert(H5HF_t *fh, size_t size, const void *obj, void *id )
{
    H5HF_hdr_t *hdr       = NULL; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(obj);
    assert(id);

    
    if (size == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "can't insert 0-sized objects");

    
    fh->hdr->f = fh->f;

    
    hdr = fh->hdr;

    
    if (size > hdr->max_man_size) {
        
        H5_WARN_CAST_AWAY_CONST_OFF
        if (H5HF__huge_insert(hdr, size, (void *)obj, id) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL, "can't store 'huge' object in fractal heap");
        H5_WARN_CAST_AWAY_CONST_ON
    } 
    
    else if (size <= hdr->tiny_max_len) {
        
        if (H5HF__tiny_insert(hdr, size, obj, id) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL, "can't store 'tiny' object in fractal heap");
    } 
    else {
        
        if (hdr->write_once) {
            HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "'write once' managed blocks not supported yet");
        } 
        else {
            
            if (H5HF__man_insert(hdr, size, obj, id) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL, "can't store 'managed' object in fractal heap");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_get_obj_len(H5HF_t *fh, const void *_id, size_t *obj_len_p)
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(id);
    assert(obj_len_p);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        if (H5HF__man_get_obj_len(fh->hdr, id, obj_len_p) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get 'managed' object's length");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        if (H5HF__huge_get_obj_len(fh->hdr, id, obj_len_p) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get 'huge' object's length");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        if (H5HF__tiny_get_obj_len(fh->hdr, id, obj_len_p) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get 'tiny' object's length");
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_get_obj_off(H5HF_t *fh, const void *_id, hsize_t *obj_off_p)
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(id);
    assert(obj_off_p);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        H5HF__man_get_obj_off(fh->hdr, id, obj_off_p);
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        
        if (H5HF__huge_get_obj_off(fh->hdr, id, obj_off_p) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get 'huge' object's offset");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        
        *obj_off_p = (hsize_t)0;
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_read(H5HF_t *fh, const void *_id, void *obj )
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(id);
    assert(obj);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        
        if (H5HF__man_read(fh->hdr, id, obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't read object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        
        if (H5HF__huge_read(fh->hdr, id, obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't read 'huge' object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        
        if (H5HF__tiny_read(fh->hdr, id, obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't read 'tiny' object from fractal heap");
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_write(H5HF_t *fh, void *_id, bool H5_ATTR_UNUSED *id_changed, const void *obj)
{
    uint8_t *id = (uint8_t *)_id; 
    uint8_t  id_flags;            
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(id);
    assert(obj);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        
        
        if (H5HF__man_write(fh->hdr, id, obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "can't write to 'managed' heap object");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        
        if (H5HF__huge_write(fh->hdr, id, obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "can't write to 'huge' heap object");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        
        
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "modifying 'tiny' object not supported yet");
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_op(H5HF_t *fh, const void *_id, H5HF_operator_t op, void *op_data)
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(id);
    assert(op);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        
        if (H5HF__man_op(fh->hdr, id, op, op_data) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "can't operate on object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        
        if (H5HF__huge_op(fh->hdr, id, op, op_data) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "can't operate on 'huge' object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        
        if (H5HF__tiny_op(fh->hdr, id, op, op_data) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "can't operate on 'tiny' object from fractal heap");
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_remove(H5HF_t *fh, const void *_id)
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(fh->hdr);
    assert(id);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    fh->hdr->f = fh->f;

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        
        if (H5HF__man_remove(fh->hdr, id) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        
        if (H5HF__huge_remove(fh->hdr, id) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove 'huge' object from fractal heap");
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        
        if (H5HF__tiny_remove(fh->hdr, id) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove 'tiny' object from fractal heap");
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_close(H5HF_t *fh)
{
    bool    pending_delete = false;       
    haddr_t heap_addr      = HADDR_UNDEF; 
    herr_t  ret_value      = SUCCEED;     

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);

    
    if (0 == H5HF__hdr_fuse_decr(fh->hdr)) {
        
        fh->hdr->f = fh->f;

        
        
        if (H5HF__space_close(fh->hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release free space info");

        
        
        if (H5HF__man_iter_ready(&fh->hdr->next_block))
            if (H5HF__man_iter_reset(&fh->hdr->next_block) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reset block iterator");

        
        
        if (H5HF__huge_term(fh->hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release 'huge' object info");

        
        if (fh->hdr->pending_delete) {
            
            pending_delete = true;
            heap_addr      = fh->hdr->heap_addr;
        } 
    }     

    
    
    if (H5HF__hdr_decr(fh->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared heap header");

    
    if (pending_delete) {
        H5HF_hdr_t *hdr; 

        
        if (NULL == (hdr = H5HF__hdr_protect(fh->f, heap_addr, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

        
        if (H5HF__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDELETE, FAIL, "unable to delete fractal heap");
    } 

done:
    
    fh = H5FL_FREE(H5HF_t, fh);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_delete(H5F_t *f, haddr_t fh_addr)
{
    H5HF_hdr_t *hdr       = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(fh_addr));

    
    if (NULL == (hdr = H5HF__hdr_protect(f, fh_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

    
    if (hdr->file_rc)
        hdr->pending_delete = true;
    else {
        
        if (H5HF__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDELETE, FAIL, "unable to delete fractal heap");
        hdr = NULL;
    } 

done:
    
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, fh_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
