#' Information for IRT Model
#'
#' @description
#' Compute test information for a unidimensional IRT model (1PL/2PL/3PL) across
#' a vector of ability values.
#'
#' @param theta Numeric vector of ability values at which to compute test
#'   information.
#' @param ip A data frame or matrix of item parameters. Columns are interpreted
#'   in order as:
#'   \itemize{
#'     \item 3 columns: \code{b}, \code{a}, \code{c} (3PL, with \code{a} on the 1.702 metric),
#'     \item 2 columns: \code{b}, \code{a} (2PL, \code{c} internally set to 0),
#'     \item 1 column: \code{b} (1PL/Rasch, \code{a = 1}, \code{c = 0}).
#'   }
#' @param est Character string indicating the estimation method:
#'   \code{"MLE"} for maximum likelihood or \code{"EAP"} for empirical Bayes.
#' @param D A numeric constant representing the scaling factor of the IRT model.
#'   Defaults to \code{1.702}.
#'
#' @details
#' Test information at each \eqn{\theta} is the sum of item information.
#' For \code{est = "EAP"}, this function returns
#' \deqn{
#'   I_{\mathrm{EAP}}(\theta) = I_{\mathrm{MLE}}(\theta) + 1,
#' }
#' where the additional 1 reflects the prior (population) contribution under a
#' standard normal prior.
#'
#' @return A list with:
#' \describe{
#'   \item{theta}{Vector of ability values.}
#'   \item{infoMLE}{If \code{est = "MLE"}, vector of test information at each \code{theta}.}
#'   \item{infoEAP}{If \code{est = "EAP"}, vector of test information at each \code{theta}.}
#' }
#'
#' @export
info <- function(theta, ip, est = c("MLE", "EAP"), D = 1.702) {

  est <- match.arg(est)

  # ---------- basic checks ----------------------------------------------------
  if (missing(theta)) {
    stop("`theta` must be supplied as a numeric vector.")
  }
  if (!is.numeric(theta) || length(theta) < 1L) {
    stop("`theta` must be a non-empty numeric vector.")
  }

  if (missing(ip)) {
    stop("`ip` must be supplied as item parameter matrix or data frame.")
  }
  if (!is.data.frame(ip) && !is.matrix(ip)) {
    stop("`ip` must be a data frame or matrix with item parameters.")
  }

  ip <- as.data.frame(ip)
  if (ncol(ip) < 1L || ncol(ip) > 3L) {
    stop("`ip` must have 1, 2, or 3 columns (b[, a[, c]]).")
  }

  # ensure numeric
  if (!all(vapply(ip, is.numeric, logical(1L)))) {
    stop("All columns in `ip` must be numeric.")
  }

  # ---------- standardize ip to 3PL form -------------------------------------
  if (ncol(ip) == 3L) {
    names(ip) <- c("b", "a", "c")
  } else if (ncol(ip) == 2L) {
    names(ip) <- c("b", "a")
    ip$c <- 0
  } else if (ncol(ip) == 1L) {
    names(ip) <- "b"
    ip$a <- 1
    ip$c <- 0
  }

  ni <- nrow(ip)
  if (ni < 1L) {
    stop("`ip` must contain at least one item.")
  }

  # ---------- replicate item params over theta grid ---------------------------
  n_theta <- length(theta)

  ip_rep <- ip[rep(seq_len(ni), times = n_theta), , drop = FALSE]
  ip_rep$theta <- rep(theta, each = ni)

  # ---------- compute item information ---------------------------------------
  ip_rep <- within(ip_rep, {
    P <- c + (1 - c) / (1 + exp(-D * a * (theta - b)))
    Q <- 1 - P
    info <- D^2 * a^2 * (Q / P) * ((P - c)^2 / (1 - c)^2)
  })

  # guard against numerical issues (e.g., P ~ 0)
  ip_rep$info[!is.finite(ip_rep$info)] <- NA_real_

  # ---------- aggregate test information by theta -----------------------------
  ip_info <- stats::aggregate(ip_rep$info,
                              by = list(theta = ip_rep$theta),
                              FUN = sum,
                              na.rm = TRUE)
  names(ip_info)[names(ip_info) == "x"] <- "infoMLE"

  # EAP: add 1 for prior contribution
  ip_info$infoEAP <- ip_info$infoMLE + 1

  # ---------- return ----------------------------------------------------------
  if (est == "MLE") {
    return(list(theta = ip_info$theta, infoMLE = ip_info$infoMLE))
  } else { # est == "EAP"
    return(list(theta = ip_info$theta, infoEAP = ip_info$infoEAP))
  }
}
