#' Estimate Standardized Regression Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @details The vector of standardized regression coefficients
#'   (\eqn{\boldsymbol{\hat{\beta}}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{\boldsymbol{\hat{\beta}}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{jackknife}{Jackknife estimates.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{fun}{Function used ("BetaNB").}
#'   }
#'
#' @param object Object of class `nb`, that is,
#'   the output of the `NB()` function.
#' @param alpha Numeric vector.
#'   Significance level \eqn{\alpha}.
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaNB")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # NB -----------------------------------------------------------------------
#' nb <- NB(
#'   object,
#'   R = 100, # use a large value e.g., 5000L for actual research
#'   seed = 0508
#' )
#'
#' # BetaNB -------------------------------------------------------------------
#' out <- BetaNB(nb, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB std
#' @export
BetaNB <- function(object,
                   alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "nb"
    )
  )
  fun <- "BetaNB"
  est <- object$lm_process$betastar
  foo <- function(x) {
    .BetaStarofSigma(
      sigmacap = x,
      q = 1 / sqrt(diag(x)),
      k = object$lm_process$k
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  out
}
