#' Simulate Poisson-Binomial Dynamic GLM
#'
#' Generates a dynamic regression dataset where each response
#' y_i | p_i ~ Binomial(n, p_i) with p_i = plogis(x_i' beta_i).
#' The latent coefficients beta_i follow a Vector-AR(1) process.
#'
#' @param N Integer > 1, number of time points/individuals.
#' @param n Integer > 0, binomial number of trials (constant across units).
#' @param q Integer > 0, number of predictors (including intercept if desired).
#'
#' @return A list with components:
#' \item{X}{N x q design matrix.}
#' \item{Y}{Integer vector of length N, counts of successes.}
#' \item{beta}{N x q matrix of dynamic regression coefficients.}
#' \item{G}{q x q autoregressive multiplier matrix (diagonal).}
#' \item{Sig}{q x q innovation covariance matrix (diagonal).}
#'
#' @examples
#' set.seed(1)
#' dat <- simPoisBin(N = 500, n = 10, q = 4)
#' head(dat$Y)
#'
#' @export
simPoisBin <- function(N = 1000L, n = 10L, q = 4L) {
  # Input checks
  if (N <= 1) stop("N must be > 1")
  if (n <= 0) stop("n must be > 0")
  if (q <= 0) stop("q must be > 0")
  
  # Generate design matrix
  X <- matrix(stats::rnorm(N * q), nrow = N, ncol = q)
  
  # Dynamic regression coefficients (VAR(1))
  beta <- matrix(NA_real_, N, q)
  G <- diag(stats::runif(q, min = 0.3, max = 0.9))   # stable AR roots
  Sig <- diag(stats::runif(q, min = 0.5, max = 2))  # innovation variances
  beta[1, ] <- stats::rnorm(q)                       # initial state
  
  for (i in 2:N) {
    beta[i, ] <- G %*% beta[i - 1, ] +
      MASS::mvrnorm(n = 1, mu = numeric(q), Sigma = Sig)
  }
  
  # Success probabilities & response
  p <- stats::plogis(rowSums(X * beta))  # element-wise product and row sum
  Y <- stats::rbinom(n = N, size = n, prob = p)
  
  list(X = X, Y = Y, beta = beta, G = G, Sig = Sig)
}
