#' @title Extract Components from Qval Package Objects
#'
#' @description 
#' A unified extractor function for retrieving internal components from objects produced by the \pkg{Qval} package. 
#' This method allows users to access key elements such as model results, validation logs, and simulation settings 
#' in a structured and object-oriented manner.
#'
#' @details 
#' This generic extractor supports three core object classes: \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, 
#' \code{\link[Qval]{sim.data}}, \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, \code{\link[Qval]{att.hierarchy}}. 
#' It is intended to streamline access to commonly used internal components without manually referencing object slots.
#' The available components for each class are listed below:
#' 
#' \describe{
#'   \item{\strong{\code{CDM}}}{Cognitive Diagnosis Model fitting results. Available components:
#'     \describe{
#'       \item{\code{analysis.obj}}{The internal model fitting object (e.g., GDINA or Baseline Model).}
#'       \item{\code{alpha}}{Estimated attribute profiles (EAP estimates) for each respondent.}
#'       \item{\code{P.alpha.Xi}}{Posterior distribution of latent attribute patterns.}
#'       \item{\code{alpha.P}}{Marginal attribute mastery probabilities (estimated).}
#'       \item{\code{P.alpha}}{Prior attribute probabilities at convergence.}
#'       \item{\code{pattern}}{The attribute mastery pattern matrix containing all possible attribute mastery pattern.}
#'       \item{\code{Deviance}}{Negative twice the marginal log-likelihood (model deviance).}
#'       \item{\code{npar}}{Number of free parameters estimated in the model.}
#'       \item{\code{AIC}}{Akaike Information Criterion.}
#'       \item{\code{BIC}}{Bayesian Information Criterion.}
#'       \item{\code{call}}{The original model-fitting function call.}
#'       \item{\code{...}}{Can \code{\link[GDINA]{extract}} corresponding value from the \code{\link[GDINA]{GDINA}} object.}
#'     }}
#'
#'   \item{\strong{\code{validation}}}{Q-matrix validation results. Available components:
#'     \describe{
#'       \item{\code{Q.orig}}{The original Q-matrix submitted for validation.}
#'       \item{\code{Q.sug}}{The suggested (revised) Q-matrix after validation.}
#'       \item{\code{time.cost}}{Total computation time for the validation procedure.}
#'       \item{\code{process}}{Log of Q-matrix modifications across iterations.}
#'       \item{\code{iter}}{Number of iterations performed during validation.}
#'       \item{\code{priority}}{Attribute priority matrix (available for PAA-based methods only).}
#'       \item{\code{Hull.fit}}{Data required to plot the Hull method results (for Hull-based validation only).}
#'       \item{\code{call}}{The original function call used for validation.}
#'     }}
#'
#'   \item{\strong{\code{sim.data}}}{Simulated data and parameters used in cognitive diagnosis simulation studies:
#'     \describe{
#'       \item{\code{dat}}{Simulated dichotomous response matrix (\eqn{N \times I}).}
#'       \item{\code{Q}}{Q-matrix used to generate the data.}
#'       \item{\code{attribute}}{True latent attribute profiles (\eqn{N \times K}).}
#'       \item{\code{catprob.parm}}{Item-category conditional success probabilities (list format).}
#'       \item{\code{delta.parm}}{Item-level delta parameters (list format).}
#'       \item{\code{higher.order.parm}}{Higher-order model parameters (if used).}
#'       \item{\code{mvnorm.parm}}{Parameters for the multivariate normal attribute distribution (if used).}
#'       \item{\code{LCprob.parm}}{Latent class-based success probability matrix.}
#'       \item{\code{call}}{The original function call that generated the simulated data.}
#'     }}
#'     
#'   \item{\strong{\code{fit}}}{Relative fit indices (-2LL, AIC, BIC, CAIC, SABIC) and absolute fit indices (\eqn{M_2} test, \eqn{RMSEA_2}, SRMSR):
#'     \describe{
#'        \item{\code{npar}}{The number of parameters.}
#'        \item{\code{-2LL}}{The Deviance.}
#'        \item{\code{AIC}}{The Akaike information criterion.}
#'        \item{\code{BIC}}{The Bayesian information criterion.}
#'        \item{\code{CAIC}}{The consistent Akaike information criterion.}
#'        \item{\code{SABIC}}{The Sample-size Adjusted BIC.}
#'        \item{\code{M2}}{A vector consisting of \eqn{M_2} statistic, degrees of freedom, significance level, and \eqn{RMSEA_2} (Liu, Tian, & Xin, 2016).}
#'        \item{\code{SRMSR}}{The standardized root mean squared residual (SRMSR; Ravand & Robitzsch, 2018).}
#'      }
#'    }
#'    
#'    \item{\strong{\code{is.Qident}}}{Results of whether the Q-matrix is identifiable:
#'      \describe{
#'        \item{\code{completeness}}{TRUE if \eqn{K \times K} identity submatrix exists.}
#'        \item{\code{distinctness}}{TRUE if remaining columns are distinct.}
#'        \item{\code{repetition}}{TRUE if every attribute appears more than 3 items.}
#'        \item{\code{genericCompleteness}}{TRUE if two different generic complete \eqn{K \times K} submatrices exist.}
#'        \item{\code{genericRepetition}}{TRUE if at least one '1' exists outside those submatrices.}
#'        \item{\code{Q1, Q2}}{Identified generic complete submatrices (if found).}
#'        \item{\code{Q.star}}{Remaining part after removing rows in \code{Q1} and \code{Q2}.}
#'        \item{\code{locallyGenericIdentifiability}}{TRUE if local generic identifiability holds.}
#'        \item{\code{globallyGenericIdentifiability}}{TRUE if global generic identifiability holds.}
#'        \item{\code{Q.reconstructed.DINA}}{Reconstructed Q-matrix with low-frequency attribute moved to first column.}
#'      }
#'    }
#'    
#'    \item{\strong{\code{att.hierarchy}}}{Results of iterative attribute hierarchy exploration:
#'      \describe{
#'        \item{\code{noSig}}{TRUE all structural parameters are not greater than 0.}
#'        \item{\code{isNonverge}}{TRUE if convergence was achieved.}
#'        \item{\code{statistic}}{A 4-column \code{data.frame} results for each structural parameter that is significantly larger than 0.}
#'        \item{\code{pattern}}{The attribute pattern matrix under iterative attribute hierarchy.}
#'      }
#'    }
#'  }
#'
#' @param object An object of class \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, 
#'               \code{\link[Qval]{sim.data}}, \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, 
#'               \code{\link[Qval]{att.hierarchy}}.
#' @param what A character string specifying the name of the component to extract.
#' @param ... Additional arguments (currently ignored).
#'
#' @return The requested component. The return type depends on the specified \code{what} and the class of the \code{object}.
#' 
#' @references
#' Khaldi, R., Chiheb, R., & Afa, A.E. (2018). Feed-forward and Recurrent Neural Networks for Time Series Forecasting: Comparative Study. In: Proceedings of the International Conference on Learning and Optimization Algorithms: Theory and Applications (LOPAL 18). Association for Computing Machinery, New York, NY, USA, Article 18, 1–6. DOI: 10.1145/3230905.3230946.
#'
#' Liu, Y., Tian, W., & Xin, T. (2016). An application of M2 statistic to evaluate the fit of cognitive diagnostic models. Journal of Educational and Behavioral Statistics, 41, 3–26. DOI: 10.3102/1076998615621293.
#'
#' Ravand, H., & Robitzsch, A. (2018). Cognitive diagnostic model of best choice: a study of reading comprehension. Educational Psychology, 38, 1255–1277. DOI: 10.1080/01443410.2018.1489524.
#' 
#' @examples
#' library(Qval)
#' set.seed(123)
#' 
#' \donttest{
#' ################################################################
#' # Example 1: sim.data extraction                               #
#' ################################################################
#' Q <- sim.Q(3, 10)
#' data.obj <- sim.data(Q, N = 200)
#' extract(data.obj, "dat")
#' 
#' 
#' ################################################################
#' # Example 2: CDM extraction                                    #
#' ################################################################
#' CDM.obj <- CDM(data.obj$dat, Q)
#' extract(CDM.obj, "alpha")
#' extract(CDM.obj, "AIC")
#' 
#' 
#' ################################################################
#' # Example 3: validation extraction                             #
#' ################################################################
#' validation.obj <- validation(data.obj$dat, Q, CDM.obj)
#' Q.sug <- extract(validation.obj, "Q.sug")
#' print(Q.sug)
#' 
#' 
#' ################################################################
#' # Example 4: fit extraction                                    #
#' ################################################################
#' fit.obj <- fit(data.obj$dat, Q.sug, model="GDINA")
#' extract(fit.obj, "M2")
#' }
#' 
#' @name extract
#' @export
extract <- function(object, what, ...) {
  UseMethod("extract")
}

#' @describeIn extract Extract fields from a \code{CDM} object
#' @importFrom GDINA extract
#' @export
extract.CDM <- function(object, what, ...) {
  qval_choices <- c("analysis.obj", "alpha", "P.alpha.Xi", "alpha.P", "P.alpha", "pattern", 
                    "Deviance", "npar", "AIC", "BIC", "call")
  if (what %in% qval_choices) {
    return(switch(what,
                  analysis.obj = object$analysis.obj,
                  alpha        = object$alpha,
                  P.alpha.Xi   = object$P.alpha.Xi,
                  alpha.P      = object$alpha.P,
                  P.alpha      = object$P.alpha,
                  pattern      = object$pattern, 
                  Deviance     = object$model.fit["Deviance"],
                  npar         = object$model.fit["npar"],
                  AIC          = object$model.fit["AIC"],
                  BIC          = object$model.fit["BIC"],
                  call         = object$call))
  }
  
  # 否则尝试从 GDINA::extract 提取（前提是 analysis.obj 为 GDINA 对象）
  if (!is.null(object$analysis.obj)) {
    return(GDINA::extract(object$analysis.obj, what = what, ...))
  }
  
  stop(sprintf("'%s' is not a valid field for CDM objects. Choose from: %s or GDINA fields.",
               what, paste(qval_choices, collapse = ", ")), call. = FALSE)
}

#' @describeIn extract Extract fields from a \code{validation} object
#' @export
extract.validation <- function(object, what, ...) {
  choices <- c("Q.orig", "Q.sug", "time.cost", "process", "iter",
               "priority", "Hull.fit", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for validation objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         Q.orig   = object$Q.orig,
         Q.sug    = object$Q.sug,
         time.cost = object$time.cost,
         process  = object$process,
         iter     = object$iter,
         priority = object$priority,
         Hull.fit = object$Hull.fit,
         call     = object$call)
}

#' @describeIn extract Extract fields from a \code{sim.data} object
#' @export
extract.sim.data <- function(object, what, ...) {
  choices <- c("dat", "Q", "attribute", "catprob.parm", "delta.parm",
               "higher.order.parm", "mvnorm.parm", "att.str", 
               "LCprob.parm", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for sim.data objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         dat                = object$dat,
         Q                  = object$Q,
         attribute          = object$attribute,
         catprob.parm       = object$catprob.parm,
         delta.parm         = object$delta.parm,
         higher.order.parm  = object$higher.order.parm,
         mvnorm.parm        = object$mvnorm.parm, 
         att.str            = object$att.str, 
         LCprob.parm        = object$LCprob.parm,
         call               = object$call)
}

#' @describeIn extract Extract fields from a \code{fit} object
#' @importFrom GDINA extract
#' @export
extract.fit <- function(object, what, ...) {
  choices <- c("npar", "-2LL", "AIC", "BIC", "CAIC", "SABIC",
               "M2", "SRMSR", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for sim.data objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         npar               = object$npar,
         `-2LL`             = object$`-2LL`,
         AIC                = object$AIC,
         BIC                = object$BIC,
         CAIC               = object$CAIC,
         SABIC              = object$SABIC,
         M2                 = object$M2,
         SRMSR              = object$SRMSR,
         call               = object$call)
}

#' @describeIn extract Extract fields from a \code{is.Qident} object
#' @export
extract.is.Qident <- function(object, what, ...) {
  choices <- c("completeness", "distinctness", "repetition", "genericCompleteness", "genericRepetitions", "Q1",
               "Q2", "Q.star", "locallyGenericIdentifiability", "globallyGenericIdentifiability", 
               "Q.reconstructed.DINA", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for sim.data objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         completeness                    = object$strictIdentifiability.obj$completeness,
         distinctness                    = object$strictIdentifiability.obj$distinctness,
         repetition                      = object$strictIdentifiability.obj$repetition,
         genericCompleteness             = object$genericIdentifiability.obj$genericCompleteness,
         genericRepetitions              = object$genericIdentifiability.obj$genericRepetition,
         Q1                              = object$genericIdentifiability.obj$Q1,
         Q2                              = object$genericIdentifiability.obj$Q2,
         Q.star                          = object$genericIdentifiability.obj$Q.star, 
         locallyGenericIdentifiability   = object$genericIdentifiability.DINA.obj$locallyGenericIdentifiability,
         globallyGenericIdentifiability  = object$genericIdentifiability.DINA.obj$globallyGenericIdentifiability, 
         Q.reconstructed.DINA            = object$genericIdentifiability.DINA.obj$Q.reconstructed.DINA, 
         call                            = object$call)
}

#' @describeIn extract Extract fields from a \code{att.hierarchy} object
#' @export
extract.att.hierarchy <- function(object, what, ...) {
  choices <- c("statistic", "pattern", "noSig", "isNonverge", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for sim.data objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         statistic  = object$statistic,
         pattern    = object$pattern,
         noSig      = object$noSig,
         isNonverge = object$isNonverge,
         call       = object$call)
}