% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_models.R
\name{fit_models}
\alias{fit_models}
\title{Fit Multiple INLA Models}
\usage{
fit_models(
  formulas,
  data,
  family,
  name,
  offset = NULL,
  control_compute = list(config = FALSE, vcov = FALSE),
  nthreads = 8,
  pb = FALSE
)
}
\arguments{
\item{formulas}{A \code{GHRformulas} object containing multiple INLA model formulas.}

\item{data}{A data frame containing the variables used in the model formulas.}

\item{family}{A character string specifying the likelihood family
(e.g., \code{"poisson"}, \code{"nbinomial"}, etc.).}

\item{name}{A character string to label each fitted model (e.g., \code{"mod"}).}

\item{offset}{A character string specifying the name of the offset variable in \code{data}.
If \code{NULL}, no offset is applied. Default is \code{NULL}.
Internally, \code{log(offset_values)} is applied.}

\item{control_compute}{A named list controlling additional computation options:
\describe{
\item{\code{config}}{Logical ; if \code{TRUE}, stores the Gaussian Markov Random Field (GMRF) and enables the
computation of posterior predictive distribution (1,000 draws). Defaults to \code{FALSE}.}
\item{\code{vcov}}{Logical if \code{TRUE}, returns  the variance-covariance
(correlation) matrix of fixed effects. Defaults to \code{FALSE}.}
}}

\item{nthreads}{An integer specifying the number of threads for parallel computation. Default is \code{8}.}

\item{pb}{Logical; if \code{TRUE}, displays a progress bar while fitting models. Default is \code{FALSE}.}
}
\value{
An object of class \code{GHRmodels} containing:
\describe{
\item{\verb{$mod_gof}}{A data frame of model-specific goodness-of-fit metrics.}
\item{\verb{$fitted}}{A list of fitted values (one element per model).
If \code{config = TRUE}, these are derived from the posterior predictive distribution (PPD);
otherwise, they are extracted from INLA's \code{summary.fitted.values}.}
\item{\verb{$fixed}}{A list of summary tables for fixed effects (one element per model).}
\item{\verb{$random}}{A list of summary tables for random effects (one element per model).}
\item{\verb{$formulas}}{A character vector of the original model formulas used.}
\item{\verb{$re}}{A character vector specifying any random effects defined in \code{formulas}.}
\item{\verb{$outcome}}{A character string indicating the outcome variable used.}
\item{\verb{$data}}{The original data frame passed to the function.}
}
}
\description{
This function fits a set of \code{INLA} model formulas, provided in a \code{GHRformulas} object,
to a specified dataset. For each fitted model, it extracts a range of outputs,
including goodness-of-fit (GoF) metrics and other model outputs (fitted values, fixed effects, random effects).
Results are extracted and stored in a \code{GHRmodels} object.
}
\details{
This function iterates over each formula in the \code{GHRformulas} object
and fits the corresponding \code{INLA} model using the internal function
\code{.fit_single_model()}. For each fitted model, it extracts the fitted values,
fixed effects, and random effects summaries. Then, it calculates a series of
model evaluation metrics using the \code{.gof_single_model()} internal function.

The goodness-of-fit (GoF) metrics are organized into two categories:

\strong{A) Model-Specific Goodness-of-Fit Metrics}

These are computed separately for each model:
\enumerate{
\item \strong{Deviance Information Criterion (DIC)}
\deqn{DIC = \bar{D} + p_{D}}
where \eqn{\bar{D}} is the posterior mean deviance and \eqn{p_{D}} is the
effective number of parameters. Lower DIC values indicate a better model fit,
balancing goodness-of-fit and model complexity.
\item \strong{Watanabe-Akaike Information Criterion (WAIC)}
\deqn{WAIC = -2\left(\mathrm{lppd} - p_{\mathrm{WAIC}}\right)}
WAIC evaluates predictive accuracy and penalizes model complexity through
the log pointwise predictive density (\eqn{\mathrm{lppd}}). Lower values
imply better generalization.
\item \strong{Log Mean Score (LMS)}
\deqn{LMS = \frac{1}{n} \sum_{i=1}^n \left( -\log(\mathrm{CPO}_i) \right)}
LMS assesses the average negative log-predictive density using Conditional Predictive
Ordinates (CPO). Lower LMS values indicate stronger predictive performance by
penalizing models that assign low probability to observed outcomes.
\item \strong{Mean Absolute Error (MAE)}
\deqn{MAE = \frac{1}{n} \sum_{i=1}^n \left| y_i - \hat{y}_i \right|}
Measures the average absolute deviation between observed values \eqn{y_i} and
predicted values \eqn{\hat{y}_i}. Lower MAE values indicate improved fit.
If \code{config = TRUE}, MAE is computed using the full posterior predictive distribution (PPD);
otherwise, it uses point estimates from INLA's \code{summary.fitted.values}.
\item \strong{Root Mean Squared Error (RMSE)}
\deqn{RMSE = \sqrt{ \frac{1}{n} \sum_{i=1}^n (y_i - \hat{y}_i)^2 }}
Captures average squared deviation between observed and predicted values.
RMSE penalizes larger errors more heavily. Lower values reflect better model fit.
If \code{config = TRUE}, RMSE uses the PPD; otherwise, it uses point estimates.
\item \strong{Continuous Ranked Probability Score (CRPS)}
\deqn{\mathrm{CRPS}(F, y) = \int_{-\infty}^{\infty} \left[F(t) - \mathbf{1}\{y \leq t\}\right]^2 dt}
CRPS assesses how well the predictive cumulative distribution aligns with the observed outcome.
Lower scores suggest better calibrated predictive distributions. Only available when \code{config = TRUE}.
}

\strong{B) Model Comparison Metrics (relative to the first model)}

The first model in the list is treated as the baseline for model comparisons. All other models
are evaluated against it using the following metrics:
\enumerate{
\item \strong{Difference in DIC and WAIC}
Stored as \code{dic_vs_first} and \code{waic_vs_first}. These represent how much higher (or lower)
each model's DIC/WAIC is compared to the first model.
Additionally, 95\% credible intervals for these differences are stored as
\code{*_vs_first_lci} and \code{*_vs_first_uci}.
\item \strong{Difference in MAE and RMSE}
Stored as \code{mae_vs_first} and \code{rmse_vs_first}. These reflect the absolute difference
in prediction error compared to the first model. No credible intervals are computed for these metrics.
\item \strong{Continuous Ranked Probability Score Skill Score (CRPSS)}
\deqn{\mathrm{CRPSS} = 1 - \frac{\mathrm{CRPS}_{\text{model}}}{\mathrm{CRPS}_{\text{baseline}}}}
Indicates how much better the predictive distribution of the current model is
relative to the baseline model. Values closer to 1 indicate improvement; negative values
imply worse performance. Available only when \code{config = TRUE}.
\item \strong{Pseudo R-squared based on deviance}
\deqn{R^2 = 1 - \exp\left( \frac{-2}{n} \left( \frac{dev_{\text{model}}}{-2} - \frac{dev_{\text{base}}}{-2} \right) \right)}
Captures relative deviance reduction compared to the baseline model. Values range from 0
(no improvement) to 1 (strong improvement).
\item \strong{Random Effect Variance}
\deqn{\mathrm{Var}_{re} = \frac{1}{\mathrm{precision}}}
Quantifies residual variance due to group- or cluster-level effects. Computed only when
random effects are defined in the model formula.
\item \strong{Proportional Change in Random Effect Variance}
\deqn{\frac{\mathrm{Var}_{re}}{\mathrm{Var}_{re}^{(1)}} - 1}
Represents the relative change in group-level variance compared to the baseline model.
Helps assess how much variance is explained by added covariates.
}
}
\examples{
\donttest{
# Load example dataset
data(dengueMS)

# Declare formulas
formulas <- c(
  "dengue_cases ~ tmin +  f(year, model='rw1')",
  "dengue_cases ~ pdsi +  f(year, model='rw1')"
)

# Tranform formulas into a 'GHRformulas' object
ghr_formulas <- as_GHRformulas(formulas)

# Fit multiple models 
results <- fit_models(
  formulas = ghr_formulas,
  data     = dengue_MS,
  family   = "nbinomial",
  name     = "TestModel",
  offset   = "population",
  nthreads = 2,
  control_compute = list(config = FALSE),
  pb       = TRUE
)

# Inspect goodness-of-fit metrics
results$mod_gof
}

}
\seealso{
\code{\link{as_GHRformulas}} converts a set of R-INLA-compatible formulas into a \code{GHRformulas} object.
}
