% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DRsurvfit.R
\name{DRsurvfit}
\alias{DRsurvfit}
\title{Doubly-robust estimation for survival outcomes in CRTs}
\usage{
DRsurvfit(
  data,
  formula,
  cens_formula = NULL,
  intv,
  id_var = NULL,
  method = c("marginal", "frailty"),
  estimand = c("SPCE", "RMTIF", "RMST"),
  trt_prob = NULL,
  variance = c("none", "jackknife"),
  fit_controls = NULL,
  verbose = FALSE
)
}
\arguments{
\item{data}{A \code{data.frame}.}

\item{formula}{Outcome model: e.g.,
  \code{Surv(time, status) ~ W1 + W2 + Z1 + Z2 + cluster(M)}.
  The left-hand side must be \code{Surv(time, status)} with
  \code{status in {0,1,2,...}} and \code{0} indicating censoring.

  The right-hand side \emph{must} include a \code{cluster(<id>)} term specifying
  the cluster id for CRTs. All other covariates may be individual- or cluster-level.}

\item{cens_formula}{Optional censoring model. If \code{NULL}, the censoring model
is built automatically from the outcome formula by:
\itemize{
  \item reusing the RHS (excluding \code{cluster()});
  \item using LHS \code{Surv(time, event == 0)};
}
If supplied, \code{cens_formula} is used as-is for all stage-specific fits,
but the DR estimating equations still use the stage-specific \code{event}
indicator as described above.}

\item{intv}{Character: name of the cluster-level treatment column (0/1),
constant within cluster.}

\item{id_var}{Character: name of the individual id column. If \code{NULL}, considered as single state.}

\item{method}{\code{"marginal"} or \code{"frailty"}.
\itemize{
  \item \code{"marginal"}: fits \code{survival::coxph} models with
        \code{cluster(<id>)} robust variance.
  \item \code{"frailty"}: fits \code{frailtyEM::emfrail} gamma-frailty models
        for outcome and censoring.
}}

\item{estimand}{\code{"SPCE"}, \code{"RMTIF"}, or \code{"RMST"}.
\itemize{
  \item \code{"SPCE"}: returns stage-specific survival arrays
        \code{S_stage_cluster} and \code{S_stage_ind} with dimensions
        \code{[time × 2 × Q]}.
  \item \code{"RMTIF"}: returns the generalized win-based restricted mean time in favor estimand
        at each event time, along with stage-wise contributions. For a binary
        status, this reduces to an RMST estimands.
  \item \code{"RMST"}: restricted mean survival time difference for the
        binary case \code{status in {0,1}}. This is a convenience alias for
        the \code{"RMTIF"} calculations when there is exactly one nonzero
        event state.
}}

\item{trt_prob}{Optional length-2 numeric vector \code{(p0, p1)} giving the
cluster-level treatment probabilities for arms 0 and 1. If \code{NULL}, they
are computed as the empirical proportion of treatment assignments
per cluster.}

\item{variance}{\code{"none"} or \code{"jackknife"} for variance estimation.}

\item{fit_controls}{Optional \code{frailtyEM::emfrail_control()} list, used only
when \code{method = "frailty"}. If \code{NULL}, default fast-fitting controls
are used (no standard errors from the frailtyEM fits are required here).}

\item{verbose}{Logical; currently unused but kept for future verbosity options.}
}
\value{
An object of class \code{"DRsurvfit"} with fields depending on
  \code{estimand}:

  \describe{
    \item{Common:}{
      \itemize{
        \item \code{method}: fitted method (\code{"marginal"} or \code{"frailty"}).
        \item \code{estimand}: requested estimand (\code{"SPCE"} or \code{"RMTIF"}).
        \item \code{trt_prob}: numeric vector \code{c(p0, p1)}.
        \item \code{event_time}: time grid:
          \itemize{
            \item SPCE: all event times including \code{0}.
            \item RMTIF: positive event times \eqn{\tau} at which the RMT-IF is evaluated.
          }
        \item \code{max_state}: maximum observed non-zero status.
        \item \code{cluster_col}: name of the cluster id column.
        \item \code{n_clusters}: number of clusters (\eqn{M}).
        \item \code{df_jackknife}: jackknife degrees of freedom (\eqn{M - 1}).
        \item \code{n_obs}: total number of observations.
        \item \code{n_events}: total number of non-censoring observations
              (\code{status != 0}).
        \item \code{cluster_trt_counts}: counts of treated and control clusters
              \code{c(n_trt0, n_trt1)} based on first row per cluster.
        \item \code{formula_outcome}: fully reconstructed outcome formula.
        \item \code{cens_formula}: final censoring formula used.
        \item \code{call}: the matched call.
        \item \code{jackknife}: logical indicating whether jackknife variance
              was computed.
      }
    }

    \item{If \code{estimand = "SPCE"}:}{
      \itemize{
        \item \code{S_stage_cluster}: 3D array \code{[time × 2 × Q]} with
              stage-specific cluster-level survival:
              \code{S_stage_cluster[ , 1, s]} = \eqn{S_1^{(s)}(t)},
              \code{S_stage_cluster[ , 2, s]} = \eqn{S_0^{(s)}(t)}.
        \item \code{S_stage_ind}: analogous individual-level survival array.
        \item \code{var_stage_cluster}: jackknife variances for
              \eqn{S_1^{(s)}(t)}, \eqn{S_0^{(s)}(t)}, and
              \eqn{S_1^{(s)}(t) - S_0^{(s)}(t)} as a 3D array
              \code{[time × 3 × Q]} with dimension names
              \code{comp = c("Var(S1)","Var(S0)","Var(S1-S0)")}, when
              \code{variance = "jackknife"}; otherwise \code{NULL}.
        \item \code{var_stage_ind}: analogous individual-level variance array.
      }
    }

    \item{If \code{estimand = "RMTIF"}:}{
      \itemize{
        \item \code{RMTIF_cluster}: matrix \code{[time × 3]} with columns
              \code{c("R1","R0","R1-R0")} giving the cluster-level RMT-IF curves
              at each event time \eqn{\tau}.
        \item \code{RMTIF_ind}: analogous individual-level RMT-IF matrix.
        \item \code{stagewise_cluster}: list of length \code{length(event_time)};
              each element is a \code{3 × (Q)} matrix of stage-wise
              contributions with rows
              \code{c("s1qs0qp1","s0qs1qp1","diff")} and columns
              \code{c("stage_1",...,"stage_Q","sum")}.
        \item \code{stagewise_ind}: analogous individual-level list.
        \item \code{var_rmtif_cluster}: jackknife variance/covariance matrix
              \code{[time × 4]} with columns
              \code{c("Var(R1)","Var(R0)","Var(R1-R0)","Cov(R1,R0)")},
              when \code{variance = "jackknife"}; otherwise \code{NULL}.
        \item \code{var_rmtif_ind}: analogous individual-level matrix.
        \item \code{S_stage_cluster}, \code{S_stage_ind}: the underlying
              stage-specific survival arrays are also returned for convenience.
      }
    }
  }
}
\description{
Fits doubly-robust estimators for cluster-randomized trials with right-censored survival outcomes,
including single-state and multi-state outcomes

The outcome is specified as \code{Surv(time, status)}, where
\code{status in {0,1,2,...,Q}} and \code{status = 0} denotes censoring.
Values \code{1,2,...,Q} are ordered states, with the largest state typically
representing an absorbing state (e.g., death).

The function supports two estimands:
\itemize{
  \item \strong{SPCE}: stage-specific survival probabilities
    \eqn{S_s(t)} for each state \eqn{s=1,\dots,S_{\max}} at all event times.
  \item \strong{RMTIF}: a generalized win-based restricted mean time in favor estimand
    constructed from the multi-state survival outcome. When \code{status} is
    binary (\code{0/1}), this reduces to an RMST estimand (evaluated on the full
    observed-time grid).
    \item \strong{RMST}: a special case of \strong{RMTIF} when
    \code{status in {0,1}} (one nonzero state). In this case the generalized
    RMT-IF reduces to a regular RMST contrast.
}

Jackknife variance is computed via leave-one-cluster-out re-fitting method
\itemize{
  \item For \code{estimand = "SPCE"}: variances of \eqn{S_{1}(t)}, \eqn{S_{0}(t)},
        and \eqn{S_{1}(t) - S_{0}(t)} at each time and state.
  \item For \code{estimand = "RMTIF"}: variances and covariance of
        \eqn{R_{1}(\tau)}, \eqn{R_{0}(\tau)}, and \eqn{R_{1}(\tau) - R_{0}(\tau)}
        at each event time \eqn{\tau}.
}

The returned object includes metadata needed for summaries and plotting:
final fitted outcome/censoring formulas, the cluster id column, number of clusters,
degrees of freedom for jackknife t-intervals (= M - 1), sample sizes, and the
cluster-level and individual-level estimators.
}
\examples{
\donttest{
data(datm)

## Multi-state RMT-IF (binary reduces to RMST-type)
fit_rmtif <- DRsurvfit(
  datm,
  Surv(time, event) ~ W1 + W2 + Z1 + Z2 + cluster(cluster),
  intv    = "trt",
  method  = "marginal",
  estimand = "RMTIF",
  variance = "none"
)
}
}
