## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 5,
  dpi = 100
)

## ----setup--------------------------------------------------------------------
library(splineplot)
library(mgcv)
library(survival)
library(splines)
library(ggplot2)

## ----data---------------------------------------------------------------------
set.seed(42)
n <- 500

# Continuous predictor
age <- rnorm(n, mean = 50, sd = 10)

# Non-linear effect
true_effect <- -0.05*(age - 50) + 0.001*(age - 50)^3/100

# Various outcomes
time_to_event <- rexp(n, rate = exp(true_effect))
event_status <- rbinom(n, 1, 0.8)
binary_outcome <- rbinom(n, 1, plogis(true_effect))
count_outcome <- rpois(n, lambda = exp(true_effect/2))
continuous_outcome <- true_effect + rnorm(n, 0, 0.5)

# Create data frame
data <- data.frame(
  age = age,
  time = time_to_event,
  status = event_status,
  binary = binary_outcome,
  count = count_outcome,
  continuous = continuous_outcome
)

## ----gam-cox------------------------------------------------------------------
# Fit GAM Cox model using weights
gam_cox <- gam(time ~ s(age),
               family = cox.ph(),
               weights = status,
               data = data)

# Create spline plot
splineplot(gam_cox, data,
          ylim = c(0.5, 2.0),
          xlab = "Age (years)",
          ylab = "Hazard Ratio")

## ----gam-logistic-------------------------------------------------------------
gam_logit <- gam(binary ~ s(age),
                 family = binomial(),
                 data = data)

splineplot(gam_logit, data,
          ylim = c(0.5, 2.0),
          ylab = "Odds Ratio")

## ----gam-poisson--------------------------------------------------------------
gam_poisson <- gam(count ~ s(age),
                   family = poisson(),
                   data = data)

splineplot(gam_poisson, data,
          ylab = "Rate Ratio")

## ----glm-ns-------------------------------------------------------------------
glm_ns <- glm(binary ~ ns(age, df = 4),
              family = binomial(),
              data = data)

splineplot(glm_ns, data,
          ylim = c(0.5, 2.0))

## ----glm-bs-------------------------------------------------------------------
glm_bs <- glm(count ~ bs(age, df = 4),
              family = poisson(),
              data = data)

splineplot(glm_bs, data)

## ----cox-ns-------------------------------------------------------------------
cox_ns <- coxph(Surv(time, status) ~ ns(age, df = 4),
                data = data)

splineplot(cox_ns, data,
          ylim = c(0.5, 2.0))

## ----custom-ref---------------------------------------------------------------
splineplot(gam_cox, data,
          refx = 45,  # Set reference at age 45
          ylim = c(0.5, 2.0))

## ----ci-styles----------------------------------------------------------------
# Ribbon style confidence intervals
splineplot(gam_logit, data,
          ribbon_ci = TRUE,
          ylim = c(0.5, 2.0))

## ----histogram----------------------------------------------------------------
splineplot(gam_cox, data,
          show_hist = FALSE,
          ylim = c(0.5, 2.0))

## ----log-scale----------------------------------------------------------------
splineplot(gam_logit, data,
          log_scale = TRUE)

## ----interaction--------------------------------------------------------------
# Add a grouping variable
data$group <- factor(sample(c("Male", "Female"), n, replace = TRUE))

# Fit model with interaction
gam_interact <- gam(time ~ s(age, by = group),
                   family = cox.ph(),
                   weights = status,
                   data = data)

# Plot shows separate curves for each group
splineplot(gam_interact, data,
          ylim = c(0.5, 2.0))

