% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitmm.R
\name{fitmm}
\alias{fitmm}
\title{Maximum Likelihood Estimation (MLE) of a k-th order Markov chain}
\usage{
fitmm(sequences, states, k = 1, init.estim = "mle")
}
\arguments{
\item{sequences}{A list of vectors representing the sequences.}

\item{states}{Vector of state space (of length s).}

\item{k}{Order of the Markov chain.}

\item{init.estim}{Optional. \code{init.estim} gives the method used to estimate
the initial distribution. The following methods are proposed:
\itemize{
\item \code{init.estim = "mle"}: the classical Maximum Likelihood Estimator
is used to estimate the initial distribution \code{init};
\item \code{init.estim = "stationary"}: the initial distribution is replaced by
the stationary distribution of the Markov chain (if the order of the
Markov chain is more than one, the transition matrix is converted
into a square block matrix in order to estimate the stationary
distribution);
\item \code{init.estim = "freq"}: the initial distribution is estimated by
taking the frequencies of the words of length \code{k} for all sequences;
\item \code{init.estim = "prod"}: \code{init} is estimated by using the product
of the frequencies of each letter (for all the sequences) in the word
of length \code{k};
\item \code{init.estim = "unif"}: the initial probability of each state is
equal to \eqn{1 / s}, with \eqn{s} the number of states.
}}
}
\value{
An object of class S3 \code{mmfit} (inheriting from the S3 class \link{mm}).
The S3 class \code{mmfit} contains:
\itemize{
\item All the attributes of the S3 class \link{mm};
\item An attribute \code{M} which is an integer giving the total length of
the set of sequences \code{sequences} (sum of all the lengths of the list
\code{sequences});
\item An attribute \code{loglik} which is a numeric value giving the value
of the log-likelihood of the specified Markov model based on the
\code{sequences};
\item An attribute \code{sequences} which is equal to the parameter
\code{sequences} of the function \code{fitmm} (i.e. a list of sequences used to
estimate the Markov model).
}
}
\description{
Maximum Likelihood Estimation of the transition matrix and
initial distribution of a k-th order Markov chain starting from one or
several sequences.
}
\details{
Let \eqn{X_1, X_2, ..., X_n} be a trajectory of length \eqn{n} of
the Markov chain \eqn{X = (X_m)_{m \in N}} of order \eqn{k = 1} with
transition matrix \eqn{p_{trans}(i,j) = P(X_{m+1} = j | X_m = i)}. The
maximum likelihood estimation of the transition matrix is
\eqn{\widehat{p_{trans}}(i,j) = \frac{N_{ij}}{N_{i.}}}, where \eqn{N_{ij}}
is the number of transitions from state \eqn{i} to state \eqn{j} and
\eqn{N_{i.}} is the number of transition from state \eqn{i} to any state.
For \eqn{k > 1} we have similar expressions.

The initial distribution of a k-th order Markov chain is defined as
\eqn{\mu_i = P(X_1 = i)}. Five methods are proposed for the estimation
of the latter :
\describe{
\item{Maximum Likelihood Estimator: }{The Maximum Likelihood Estimator
for the initial distribution. The formula is:
\eqn{\widehat{\mu_i} = \frac{Nstart_i}{L}}, where \eqn{Nstart_i} is
the number of occurences of the word \eqn{i} (of length \eqn{k}) at
the beginning of each sequence and \eqn{L} is the number of sequences.
This estimator is reliable when the number of sequences \eqn{L} is high.}
\item{Stationary distribution: }{The stationary distribution is used as
a surrogate of the initial distribution. If the order of the Markov
chain is more than one, the transition matrix is converted into a
square block matrix in order to estimate the stationary distribution.
This method may take time if the order of the Markov chain is high
(more than three (3)).}
\item{Frequencies of each word: }{The initial distribution is estimated
by taking the frequencies of the words of length \code{k} for all sequences.
The formula is \eqn{\widehat{\mu_i} = \frac{N_i}{N}}, where \eqn{N_i}
is the number of occurences of the word \eqn{i} (of length \eqn{k}) in
the sequences and \eqn{N} is the sum of the lengths of the sequences.}
\item{Product of the frequencies of each state: }{The initial distribution
is estimated by using the product of the frequencies of each state
(for all the sequences) in the word of length \eqn{k}.}
\item{Uniform distribution: }{The initial probability of each state is
equal to \eqn{1 / s}, with \eqn{s}, the number of states.}
}
}
\examples{
states <- c("a", "c", "g", "t")
s <- length(states)
k <- 2
init <- rep.int(1 / s ^ k, s ^ k)
p <- matrix(0.25, nrow = s ^ k, ncol = s)

# Specify a Markov model of order 2
markov <- mm(states = states, init = init, ptrans = p, k = k)

seqs <- simulate(object = markov, nsim = c(1000, 10000, 2000), seed = 150)

est <- fitmm(sequences = seqs, states = states, k = 2)

}
\seealso{
\link{mm}, \link{simulate.mm}
}
