% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lazyarray.R
\name{lazyarray}
\alias{lazyarray}
\title{Create or load 'lazyarray' instance}
\usage{
lazyarray(
  path,
  storage_format,
  dim,
  dimnames = NULL,
  multipart = TRUE,
  prefix = "",
  multipart_mode = 1,
  compress_level = 50L,
  file_names = list("", seq_len(dim[[length(dim)]]))[[multipart + 1]],
  meta_name = "lazyarray.meta",
  read_only = FALSE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{path}{path to a local drive where array data is stored}

\item{storage_format}{data type, choices are \code{"double"}, 
\code{"integer"}, \code{"character"}, and \code{"complex"}; see details}

\item{dim}{integer vector, dimension of array, see \code{\link{dim}}}

\item{dimnames}{list of vectors, names of each dimension, see \code{\link{dimnames}}}

\item{multipart}{whether to split array into multiple partitions, default is true}

\item{prefix}{character prefix of array partition}

\item{multipart_mode}{1, or 2, mode of partition, see \code{\link{create_lazyarray}}}

\item{compress_level}{0 to 100, level of compression. 0 means
no compression, 100 means maximum compression. For persistent data,
it's recommended to set 100. Default is 50.}

\item{file_names}{partition names without prefix nor extension; see details}

\item{meta_name}{header file name, default is \code{"lazyarray.meta"}}

\item{read_only}{whether created array is read-only}

\item{quiet}{whether to suppress messages, default is false}

\item{...}{ignored}
}
\description{
If path is missing, create a new array. If path exists and 
meta file is complete, load existing file, otherwise create new meta file
and import from existing data.
}
\details{
There are three cases and \code{lazyarray} behaves differently 
under each cases. Case 1: if \code{path} is missing, then the function calls 
\code{\link{create_lazyarray}} to create a blank array instance. Case 2: 
if \code{path} exists and it contains \code{meta_name}, then load existing
instance with given read/write access. In this case, parameters other than
\code{read_only}, \code{path}, \code{meta_name} will be ignored. Case 3: if 
\code{meta_name} is missing and \code{path} is missing, then \code{lazyarray}
will try to create arrays from existing data files. 

If \code{lazyarray} enters case 3, then \code{file_names} will be used to 
locate partition files. Under multi-part mode (\code{multipart=TRUE}), 
\code{file_names} is default to 1, 2, ..., \code{dim[length(dim)]}. These
correspond to \code{'1.fst'}, \code{'2.fst'}, etc. under \code{path} folder.
You may specify your own \code{file_names} if irregular names are used. 
and file format for each partition will be \code{<prefix><file_name>.fst}.
For example, a file name \code{file_names=c('A', 'B')} and 
\code{prefix="file-"} means the first partition will be stored as 
\code{"file-A.fst"}, and \code{"file-B.fst"}. It's fine if some files are
missing, the corresponding partition will be filled with \code{NA} when
trying to obtain values from those partition. However, length of 
\code{file_names} must equals to the last dimension when 
\code{multipart=TRUE}. If \code{multipart=FALSE}, \code{file_names} should 
have length 1 and the corresponding file is the data file.

It's worth note to import from existing partition files generated by 
other packages such as \code{'fst'}, the partition files must be homogeneous,
meaning the stored data length, dimension, and storage type must be the same.
Because \code{'fstcore'} package stores data in data frame internally, 
the column name must be 'V1', 'V2', etc. for non-complex elements or 
'V1R', 'V1I', ... for complex numbers (real and imaginary data are stored
in different columns).
}
\examples{

path <- tempfile()

# ---------------- case 1: Create new array ------------------
arr <- lazyarray(path, storage_format = 'double', dim = c(2,3,4), 
                 meta_name = 'lazyarray.meta')
arr[] <- 1:24

# Subset and get the first partition
arr[,,1]

# Partition file path (total 4 partitions)
arr$get_partition_fpath()

# Removing array doesn't clear the data
rm(arr); gc()

# ---------------- Case 2: Load from existing directory ----------------
## Important!!! Run case 1 first
# Load from existing path, no need to specify other params
arr <- lazyarray(path, meta_name = 'lazyarray.meta', read_only = TRUE)

arr[,,1]

# ---------------- Case 3: Import from existing data ----------------
## Important!!! Run case 1 first

# path exists, but meta is missing, all other params are required
# Notice the partition count increased from 4 to 5, and storage type converts
# from double to character
arr <- lazyarray(path = path, meta_name = 'lazyarray-character.meta', 
                 file_names = c(1,2,3,4,'additional'), 
                 storage_format = 'character', dim = c(2,3,5), 
                 quiet = TRUE, read_only = FALSE)

# partition names
arr$get_partition_fpath(1:4, full_path = FALSE)
arr$get_partition_fpath(5, full_path = FALSE)

# The first dimension still exist and valid
arr[,,1]

# The additional partition is all NA
arr[,,5]

# Set data to 5th partition
arr[,,5] <- rep(0, 6)

# -------- Advanced usage: create fst data and import manually --------

# Clear existing files
path <- tempfile()
unlink(path, recursive = TRUE)
dir.create(path, recursive = TRUE)

# Create array of dimension 2x3x4, but 3rd partition is missing
# without using lazyarray package 

# Column names must be V1 or V1R, V1I (complex)
fst::write_fst(data.frame(V1 = 1:6), path = file.path(path, 'part-1.fst'))
fst::write_fst(data.frame(V1 = 7:12), path = file.path(path, 'part-B.fst'))
fst::write_fst(data.frame(V1 = 19:24), path = file.path(path, 'part-d.fst'))

# Import via lazyarray
arr <- lazyarray(path, meta_name = 'test-int.meta',
                 storage_format = 'integer',
                 dim = c(2,3,4), prefix = 'part-', 
                 file_names = c('1', 'B', 'C', 'd'), 
                 quiet = TRUE)

arr[]

# Complex case
fst::write_fst(data.frame(V1R = 1:6, V1I = 1:6), 
               path = file.path(path, 'cplx-1.fst'))
fst::write_fst(data.frame(V1R = 7:12, V1I = 100:105), 
               path = file.path(path, 'cplx-2.fst'))
fst::write_fst(data.frame(V1R = 19:24, V1I = rep(0,6)), 
               path = file.path(path, 'cplx-4.fst'))
arr <- lazyarray(path, meta_name = 'test-cplx.meta',
                 storage_format = 'complex',
                 dim = c(2,3,4), prefix = 'cplx-', 
                 file_names = 1:4, quiet = TRUE)

arr[]

}
\seealso{
\code{\link{create_lazyarray}}, \code{\link{load_lazyarray}}
}
\author{
Zhengjia Wang
}
