% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{MockJurors}
\alias{MockJurors}
\title{Confidence of Mock Jurors in Their Verdicts}
\format{
A data frame with 104 observations on 3 variables:
\describe{
\item{confidence}{numeric. Juror confidence in their verdict, scaled to the
open unit interval (0, 1). Original scale was 0-100.}
\item{verdict}{factor indicating whether a two-option verdict (guilty vs.
acquittal) or three-option verdict (with Scottish 'not proven' alternative)
was requested. Sum contrast coding is employed.}
\item{conflict}{factor. Is there conflicting testimonial evidence? Values
are \code{no} or \code{yes}. Sum contrast coding is employed.}
}
}
\source{
Data collected by Deady (2004), analyzed by Smithson and Verkuilen (2006).
}
\usage{
MockJurors
}
\description{
Data from a study examining factors that influence mock juror confidence in
verdicts for criminal trials. The experiment manipulates verdict options
(two-option vs. three-option) and presence of conflicting testimonial evidence.
}
\details{
The data were collected by Deady (2004) among first-year psychology students at
Australian National University. The experiment examined how the availability of
a third verdict option ('not proven') and conflicting evidence affect juror
confidence.

Smithson and Verkuilen (2006) employed the data, scaling the original confidence
(on a scale 0-100) to the open unit interval using the transformation:
\code{((original_confidence/100) * 103 - 0.5) / 104}.

\strong{Important note:} The original coding of \code{conflict} in the data provided from
Smithson's homepage is -1/1 which Smithson and Verkuilen (2006) describe to
mean no/yes. However, all their results (sample statistics, histograms, etc.)
suggest that it actually means yes/no, which was employed in the corrected
\code{MockJurors} dataset.
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(MockJurors)

# Example 1: Main effects model with heteroscedasticity
# Confidence depends on verdict options and conflicting evidence
# Variability may also depend on these factors
fit_kw <- gkwreg(
  confidence ~ verdict + conflict |
    verdict * conflict,
  data = MockJurors,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha (mean): Additive effects of verdict type and conflict
#   Three-option verdicts may reduce confidence
#   Conflicting evidence reduces confidence
# - Beta (precision): Interaction suggests confidence variability
#   depends on combination of verdict options and evidence type

# Example 2: Full interaction in mean model
fit_kw_interact <- gkwreg(
  confidence ~ verdict * conflict |
    verdict * conflict,
  data = MockJurors,
  family = "kw"
)
summary(fit_kw_interact)

# Interpretation:
# - Full interaction: Third verdict option may have different effects
#   depending on whether evidence is conflicting

# Test interaction significance
anova(fit_kw, fit_kw_interact)

# Example 3: McDonald distribution for extreme confidence patterns
# Jurors may show very high confidence (ceiling effects) or very low
# confidence depending on conditions
fit_mc <- gkwreg(
  confidence ~ verdict * conflict | # gamma: full interaction
    verdict * conflict | # delta: full interaction
    verdict + conflict, # lambda: additive extremity effects
  data = MockJurors,
  family = "mc",
  control = gkw_control(
    method = "BFGS",
    maxit = 1500,
    reltol = 1e-8
  )
)
summary(fit_mc)

# Interpretation:
# - Lambda: Models asymmetry and extreme confidence
#   Some conditions produce more polarized confidence (very high or very low)

# Example 4: Exponentiated Kumaraswamy alternative
fit_ekw <- gkwreg(
  confidence ~ verdict * conflict | # alpha
    verdict + conflict | # beta
    conflict, # lambda: conflict affects extremity
  data = MockJurors,
  family = "ekw",
  control = gkw_control(
    method = "BFGS",
    maxit = 1500
  )
)
summary(fit_ekw)

# Compare 3-parameter models
AIC(fit_ekw, fit_mc)
}
}
\references{
Deady, S. (2004). \emph{The Psychological Third Verdict: 'Not Proven' or
'Not Willing to Make a Decision'?} Unpublished honors thesis, The Australian
National University, Canberra.

Smithson, M., and Verkuilen, J. (2006). A Better Lemon Squeezer?
Maximum-Likelihood Regression with Beta-Distributed Dependent Variables.
\emph{Psychological Methods}, \strong{11}(1), 54--71.
}
\keyword{datasets}
