% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{retinal}
\alias{retinal}
\title{Intraocular Gas Decay in Retinal Surgery}
\format{
A data frame with 40 observations on 7 variables:
\describe{
\item{ID}{integer. Patient identification number for longitudinal tracking.}
\item{Gas}{numeric. Proportion of intraocular gas remaining (0-1 scale).
Response variable measuring the fraction of perfluoropropane gas still
present in the vitreous cavity.}
\item{Time}{numeric. Time point of measurement (days or weeks post-surgery).}
\item{LogT}{numeric. Logarithm of time, log(Time). Used to linearize the
exponential decay pattern.}
\item{LogT2}{numeric. Squared logarithm of time, (log(Time))^2. Captures
nonlinear decay patterns.}
\item{Level}{factor. Initial gas concentration level at the time of injection.
Different starting concentrations affect decay kinetics.}
}
}
\source{
Based on clinical data from vitreoretinal surgery patients. Originally analyzed
in Song and Tan (2000).
}
\usage{
retinal
}
\description{
Longitudinal data on the recorded decay of intraocular gas (perfluoropropane)
in complex retinal surgeries. The dataset tracks the proportion of gas remaining
over time following vitrectomy procedures.
}
\details{
This longitudinal dataset comes from a study of gas decay following vitreoretinal
surgery. Perfluoropropane (C3F8) is commonly used as a temporary tamponade agent
in retinal detachment repair and other complex vitreoretinal procedures.

\strong{Clinical background:} During vitrectomy for retinal detachment, gas bubbles
are injected into the vitreous cavity to help reattach the retina by providing
internal tamponade. The gas gradually absorbs and dissipates over time. Understanding
the decay rate is important for:
\itemize{
\item Predicting when patients can resume normal activities (esp. air travel)
\item Assessing treatment efficacy
\item Planning follow-up examinations
}

\strong{Decay kinetics:} Gas decay typically follows a nonlinear pattern that can be
approximated by exponential or power-law functions. The log transformation
(LogT, LogT2) helps linearize these relationships for regression modeling.

\strong{Data structure:} This is a longitudinal/panel dataset with repeated measurements
on the same patients over time. Correlation structures (exchangeable, AR(1), etc.)
should be considered when modeling.

The proportional nature of the gas variable (bounded between 0 and 1) makes this
dataset ideal for:
\itemize{
\item Simplex marginal models (original application by Song & Tan 2000)
\item Beta regression with longitudinal correlation structures
\item Kumaraswamy regression with heteroscedastic errors
}
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(retinal)

# Example 1: Nonlinear time decay with level effects
# Model gas decay as quadratic function of log-time
# Allow precision to vary by initial gas concentration
fit_kw <- gkwreg(
  Gas ~ LogT + LogT2 + Level |
    Level,
  data = retinal,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Decay curve shape varies by initial gas concentration
#   LogT + LogT2 capture nonlinear exponential-like decay
# - Beta: Precision differs by concentration level
#   Higher concentration may produce more/less variable decay

# Example 2: Heteroscedastic model
# Variability in gas proportion may change over time
fit_kw_hetero <- gkwreg(
  Gas ~ LogT + LogT2 + Level |
    Level + LogT,
  data = retinal,
  family = "kw"
)
summary(fit_kw_hetero)

# Interpretation:
# - Beta: Precision varies with both level and time
#   Early measurements may be more variable than late measurements

# Test heteroscedasticity
anova(fit_kw, fit_kw_hetero)

# Example 3: Exponentiated Kumaraswamy for decay tails
# Gas decay may show different tail behavior at extreme time points
# (very fast initial decay or very slow residual decay)
fit_ekw <- gkwreg(
  Gas ~ LogT + LogT2 + Level | # alpha: decay curve
    Level + LogT | # beta: heteroscedasticity
    Level, # lambda: tail heaviness by level
  data = retinal,
  family = "ekw"
)
summary(fit_ekw)

# Interpretation:
# - Lambda varies by level: Different initial concentrations may have
#   different rates of extreme decay (very fast or very slow residual gas)
# - Important for predicting complete absorption time

# Example 4: McDonald distribution for asymmetric decay
# Alternative parameterization for skewed decay patterns
fit_mc <- gkwreg(
  Gas ~ LogT + LogT2 + Level | # gamma
    LogT + Level | # delta
    Level, # lambda
  data = retinal,
  family = "mc",
  control = gkw_control(
    method = "BFGS",
    maxit = 1500,
    reltol = 1e-8
  )
)
summary(fit_mc)

# Model comparison
AIC(fit_kw, fit_kw_hetero, fit_ekw, fit_mc)
}
}
\references{
Meyers, S.M., Ambler, J.S., Tan, M., Werner, J.C., and Huang, S.S. (1992).
Variation of Perfluoropropane Disappearance After Vitrectomy. \emph{Retina},
\strong{12}, 359--363.

Song, P.X.-K., and Tan, M. (2000). Marginal Models for Longitudinal Continuous
Proportional Data. \emph{Biometrics}, \strong{56}, 496--502.
\doi{10.1111/j.0006-341x.2000.00496.x}

Song, P.X.-K., Qiu, Z., and Tan, M. (2004). Modelling Heterogeneous Dispersion
in Marginal Models for Longitudinal Proportional Data. \emph{Biometrical Journal},
\strong{46}, 540--553.

Qiu, Z., Song, P.X.-K., and Tan, M. (2008). Simplex Mixed-Effects Models for
Longitudinal Proportional Data. \emph{Scandinavian Journal of Statistics},
\strong{35}, 577--596.
\doi{10.1111/j.1467-9469.2008.00603.x}

Zhang, P., Qiu, Z., and Shi, C. (2016). simplexreg: An R Package for
Regression Analysis of Proportional Data Using the Simplex Distribution.
\emph{Journal of Statistical Software}, \strong{71}(11), 1--21.
\doi{10.18637/jss.v071.i11}
}
\keyword{datasets}
