% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obwoe.R
\name{obwoe}
\alias{obwoe}
\title{Unified Optimal Binning and Weight of Evidence Transformation}
\usage{
obwoe(
  data,
  target,
  feature = NULL,
  min_bins = 2,
  max_bins = 7,
  algorithm = "auto",
  control = control.obwoe()
)
}
\arguments{
\item{data}{A \code{data.frame} containing the predictor variables (features)
and the response variable (target). All features to be binned must be present
in this data frame. The data frame should not contain list-columns.}

\item{target}{Character string specifying the column name of the response
variable. Must be a binary outcome encoded as integers \code{0} (non-event)
and \code{1} (event), or a multinomial outcome encoded as integers
\code{0, 1, 2, ..., K}. Missing values in the target are not permitted.}

\item{feature}{Optional character vector specifying which columns to process.
If \code{NULL} (default), all columns except \code{target} are processed.
Features containing only missing values are automatically skipped with a warning.}

\item{min_bins}{Integer specifying the minimum number of bins. Must satisfy
\eqn{2 \le} \code{min_bins} \eqn{\le} \code{max_bins}. Algorithms may
produce fewer bins if the data has insufficient unique values. Default is 2.}

\item{max_bins}{Integer specifying the maximum number of bins. Controls the
granularity of discretization. Higher values capture more detail but risk
overfitting. Typical values range from 5 to 10 for credit scoring applications.
Default is 7.}

\item{algorithm}{Character string specifying the binning algorithm.
Use \code{"auto"} (default) for automatic selection based on target type:
\code{"jedi"} for binary targets, \code{"jedi_mwoe"} for multinomial.
See Details for the complete algorithm taxonomy.}

\item{control}{A list of algorithm-specific control parameters created by
\code{\link{control.obwoe}}. Provides fine-grained control over convergence
thresholds, bin cutoffs, and other optimization parameters.}
}
\value{
An S3 object of class \code{"obwoe"} containing:
\describe{
  \item{\code{results}}{Named list where each element contains the binning
    result for a single feature, including:
    \describe{
      \item{\code{bin}}{Character vector of bin labels/intervals}
      \item{\code{woe}}{Numeric vector of Weight of Evidence per bin}
      \item{\code{iv}}{Numeric vector of Information Value contribution per bin}
      \item{\code{count}}{Integer vector of observation counts per bin}
      \item{\code{count_pos}}{Integer vector of positive (event) counts per bin}
      \item{\code{count_neg}}{Integer vector of negative (non-event) counts per bin}
      \item{\code{cutpoints}}{Numeric vector of bin boundaries (numerical only)}
      \item{\code{converged}}{Logical indicating algorithm convergence}
      \item{\code{iterations}}{Integer count of optimization iterations}
    }
  }
  \item{\code{summary}}{Data frame with one row per feature containing:
    \code{feature} (name), \code{type} (numerical/categorical),
    \code{algorithm} (used), \code{n_bins} (count), \code{total_iv} (sum),
    \code{error} (logical flag)}
  \item{\code{target}}{Name of the target column}
  \item{\code{target_type}}{Detected type: \code{"binary"} or \code{"multinomial"}}
  \item{\code{n_features}}{Number of features processed}
  \item{\code{call}}{The matched function call for reproducibility}
}
}
\description{
Master interface for optimal discretization and Weight of Evidence (WoE)
computation across numerical and categorical predictors. This function serves
as the primary entry point for the \strong{OptimalBinningWoE} package, providing
automatic feature type detection, intelligent algorithm selection, and unified
output structures for seamless integration into credit scoring and predictive
modeling workflows.
}
\details{
\subsection{Theoretical Foundation}{

Weight of Evidence (WoE) transformation is a staple of credit scoring methodology,
originating from information theory and the concept of evidential support
(Good, 1950; Kullback, 1959). For a bin \eqn{i}, the WoE is defined as:

\deqn{WoE_i = \ln\left(\frac{p_i}{n_i}\right) = \ln\left(\frac{N_{i,1}/N_1}{N_{i,0}/N_0}\right)}

where:
\itemize{
  \item \eqn{N_{i,1}} = number of events (target=1) in bin \eqn{i}
  \item \eqn{N_{i,0}} = number of non-events (target=0) in bin \eqn{i}
  \item \eqn{N_1}, \eqn{N_0} = total events and non-events, respectively
  \item \eqn{p_i = N_{i,1}/N_1} = proportion of events in bin \eqn{i}
  \item \eqn{n_i = N_{i,0}/N_0} = proportion of non-events in bin \eqn{i}
}

The Information Value (IV) quantifies the total predictive power of a binning:

\deqn{IV = \sum_{i=1}^{k} (p_i - n_i) \times WoE_i = \sum_{i=1}^{k} (p_i - n_i) \times \ln\left(\frac{p_i}{n_i}\right)}

where \eqn{k} is the number of bins. IV is equivalent to the Kullback-Leibler
divergence between the event and non-event distributions.
}

\subsection{Algorithm Taxonomy}{

The package provides 28 algorithms organized by supported feature types:

\strong{Universal Algorithms} (both numerical and categorical):
\tabular{lll}{
  \strong{ID} \tab \strong{Full Name} \tab \strong{Method} \cr
  \code{jedi} \tab Joint Entropy-Driven Information \tab Heuristic + IV optimization \cr
  \code{jedi_mwoe} \tab JEDI Multinomial WoE \tab Extension for K>2 classes \cr
  \code{cm} \tab ChiMerge \tab Bottom-up chi-squared merging \cr
  \code{dp} \tab Dynamic Programming \tab Exact optimal IV partitioning \cr
  \code{dmiv} \tab Decision Tree MIV \tab Recursive partitioning \cr
  \code{fetb} \tab Fisher's Exact Test \tab Statistical significance-based \cr
  \code{mob} \tab Monotonic Optimal Binning \tab IV-optimal with monotonicity \cr
  \code{sketch} \tab Sketching \tab Probabilistic data structures \cr
  \code{udt} \tab Unsupervised Decision Tree \tab Entropy-based without target
}

\strong{Numerical-Only Algorithms}:
\tabular{ll}{
  \strong{ID} \tab \strong{Description} \cr
  \code{bb} \tab Branch and Bound (exact search) \cr
  \code{ewb} \tab Equal Width Binning (unsupervised) \cr
  \code{fast_mdlp} \tab Fast MDLP with pruning \cr
  \code{ir} \tab Isotonic Regression \cr
  \code{kmb} \tab K-Means Binning \cr
  \code{ldb} \tab Local Density Binning \cr
  \code{lpdb} \tab Local Polynomial Density \cr
  \code{mblp} \tab Monotonic Binning LP \cr
  \code{mdlp} \tab Minimum Description Length \cr
  \code{mrblp} \tab Monotonic Regression LP \cr
  \code{oslp} \tab Optimal Supervised LP \cr
  \code{ubsd} \tab Unsupervised Std-Dev Based
}

\strong{Categorical-Only Algorithms}:
\tabular{ll}{
  \strong{ID} \tab \strong{Description} \cr
  \code{gmb} \tab Greedy Monotonic Binning \cr
  \code{ivb} \tab Information Value DP (exact) \cr
  \code{mba} \tab Modified Binning Algorithm \cr
  \code{milp} \tab Mixed Integer LP \cr
  \code{sab} \tab Simulated Annealing \cr
  \code{sblp} \tab Similarity-Based LP \cr
  \code{swb} \tab Sliding Window Binning
}
}

\subsection{Automatic Type Detection}{

Feature types are detected as follows:
\itemize{
  \item \strong{Numerical}: \code{numeric} or \code{integer} vectors not of class \code{factor}
  \item \strong{Categorical}: \code{character}, \code{factor}, or \code{logical} vectors
}

When \code{algorithm = "auto"}, the function selects:
\itemize{
  \item \code{"jedi"} for binary targets (recommended for most use cases)
  \item \code{"jedi_mwoe"} for multinomial targets (K > 2 classes)
}
}

\subsection{IV Interpretation Guidelines}{

Siddiqi (2006) provides the following IV thresholds for variable selection:
\tabular{ll}{
  \strong{IV Range} \tab \strong{Predictive Power} \cr
  < 0.02 \tab Unpredictive \cr
  0.02 - 0.10 \tab Weak \cr
  0.10 - 0.30 \tab Medium \cr
  0.30 - 0.50 \tab Strong \cr
  > 0.50 \tab Suspicious (likely overfitting)
}
}

\subsection{Computational Considerations}{

Time complexity varies by algorithm:
\itemize{
  \item \strong{JEDI, ChiMerge, MOB}: \eqn{O(n \log n + k^2 m)} where \eqn{n} = observations, \eqn{k} = bins, \eqn{m} = iterations
  \item \strong{Dynamic Programming}: \eqn{O(n \cdot k^2)} for exact solution
  \item \strong{Equal Width}: \eqn{O(n)} (fastest, but unsupervised)
  \item \strong{MILP, SBLP}: Potentially exponential (NP-hard problems)
}

For large datasets (\eqn{n > 10^6}), consider:
\enumerate{
  \item Using \code{algorithm = "sketch"} for approximate streaming
  \item Reducing \code{max_n_prebins} via \code{control.obwoe()}
  \item Sampling the data before binning
}
}
}
\examples{
\donttest{
# =============================================================================
# Example 1: Basic Usage with Mixed Feature Types
# =============================================================================
set.seed(42)
n <- 2000

# Simulate credit scoring data
df <- data.frame(
  # Numerical features
  age = pmax(18, pmin(80, rnorm(n, 45, 15))),
  income = exp(rnorm(n, 10, 0.8)),
  debt_ratio = rbeta(n, 2, 5),
  credit_history_months = rpois(n, 60),

  # Categorical features
  education = sample(c("High School", "Bachelor", "Master", "PhD"),
    n,
    replace = TRUE, prob = c(0.35, 0.40, 0.20, 0.05)
  ),
  employment = sample(c("Employed", "Self-Employed", "Unemployed", "Retired"),
    n,
    replace = TRUE, prob = c(0.60, 0.20, 0.10, 0.10)
  ),

  # Binary target (default probability varies by features)
  target = rbinom(n, 1, 0.15)
)

# Process all features with automatic algorithm selection
result <- obwoe(df, target = "target")
print(result)

# View detailed summary
print(result$summary)

# Access results for a specific feature
age_bins <- result$results$age
print(data.frame(
  bin = age_bins$bin,
  woe = round(age_bins$woe, 3),
  iv = round(age_bins$iv, 4),
  count = age_bins$count
))

# =============================================================================
# Example 2: Using a Specific Algorithm
# =============================================================================

# Use MDLP for numerical features (entropy-based)
result_mdlp <- obwoe(df,
  target = "target",
  feature = c("age", "income"),
  algorithm = "mdlp",
  min_bins = 3,
  max_bins = 6
)

cat("\nMDLP Results:\n")
print(result_mdlp$summary)

# =============================================================================
# Example 3: Custom Control Parameters
# =============================================================================

# Fine-tune algorithm behavior
ctrl <- control.obwoe(
  bin_cutoff = 0.02, # Minimum 2\% per bin
  max_n_prebins = 30, # Allow more initial bins
  convergence_threshold = 1e-8
)

result_custom <- obwoe(df,
  target = "target",
  feature = "debt_ratio",
  algorithm = "jedi",
  control = ctrl
)

cat("\nCustom JEDI Result:\n")
print(result_custom$results$debt_ratio$bin)

# =============================================================================
# Example 4: Comparing Multiple Algorithms
# =============================================================================

algorithms <- c("jedi", "mdlp", "ewb", "mob")
iv_comparison <- sapply(algorithms, function(algo) {
  tryCatch(
    {
      res <- obwoe(df, target = "target", feature = "income", algorithm = algo)
      res$summary$total_iv
    },
    error = function(e) NA_real_
  )
})

cat("\nAlgorithm Comparison (IV for 'income'):\n")
print(sort(iv_comparison, decreasing = TRUE))

# =============================================================================
# Example 5: Feature Selection Based on IV
# =============================================================================

# Process all features and select those with IV > 0.02
result_all <- obwoe(df, target = "target")

strong_features <- result_all$summary[
  result_all$summary$total_iv >= 0.02 & !result_all$summary$error,
  c("feature", "total_iv", "n_bins")
]
strong_features <- strong_features[order(-strong_features$total_iv), ]

cat("\nFeatures with IV >= 0.02 (predictive):\n")
print(strong_features)

# =============================================================================
# Example 6: Handling Algorithm Compatibility
# =============================================================================

# MDLP only works for numerical - will fail for categorical
result_mixed <- obwoe(df,
  target = "target",
  algorithm = "mdlp"
)

# Check for errors
cat("\nCompatibility check:\n")
print(result_mixed$summary[, c("feature", "type", "error")])
}

}
\references{
Good, I. J. (1950). Probability and the Weighing of Evidence.
\emph{Griffin, London}.

Kullback, S. (1959). Information Theory and Statistics.
\emph{Wiley, New York}.

Siddiqi, N. (2006). Credit Risk Scorecards: Developing and Implementing
Intelligent Credit Scoring. \emph{John Wiley & Sons}.
\doi{10.1002/9781119201731}

Thomas, L. C., Edelman, D. B., & Crook, J. N. (2002). Credit Scoring
and Its Applications. \emph{SIAM Monographs on Mathematical Modeling
and Computation}. \doi{10.1137/1.9780898718317}

Navas-Palencia, G. (2020). Optimal Binning: Mathematical Programming
Formulation and Solution Approach. \emph{Expert Systems with Applications},
158, 113508. \doi{10.1016/j.eswa.2020.113508}

Zeng, G. (2014). A Necessary Condition for a Good Binning Algorithm
in Credit Scoring. \emph{Applied Mathematical Sciences}, 8(65), 3229-3242.
}
\seealso{
\code{\link{control.obwoe}} for algorithm-specific parameters,
\code{\link{obwoe_algorithms}} to list all available algorithms with capabilities,
\code{\link{print.obwoe}} for display methods,
\code{\link{ob_apply_woe_num}} and \code{\link{ob_apply_woe_cat}} to apply
WoE transformations to new data.

For individual algorithms with full parameter control:
\code{\link{ob_numerical_jedi}}, \code{\link{ob_categorical_jedi}},
\code{\link{ob_numerical_mdlp}}, \code{\link{ob_categorical_ivb}}.
}
