## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 5
)

## ----eval=FALSE---------------------------------------------------------------
# # From CRAN (once available)
# install.packages("SpuriousMemory")

## -----------------------------------------------------------------------------
library(SpuriousMemory)

## -----------------------------------------------------------------------------
data(RV5min)
n <- length(RV5min)
cat("Sample size:", n, "\n")
cat("Mean:", round(mean(RV5min), 4), "\n")
cat("Std. dev:", round(sd(RV5min), 4), "\n")

## ----fig.width=7, fig.height=4------------------------------------------------
plot(RV5min, type = "l", 
     main = "Realized Volatility: Japanese Yen/US Dollar",
     xlab = "Time (days)", ylab = "Log RV", 
     col = "steelblue", lwd = 0.8)
grid()

## -----------------------------------------------------------------------------
result <- Longmemorytest(log(RV5min), demean = TRUE, alpha = 0.05)

## -----------------------------------------------------------------------------
# Local Whittle estimates
result$d_estimate

# Test statistics
result$test_stat_eps02  # W statistic (ε = 0.02)
result$test_stat_eps05  # W statistic (ε = 0.05)

# Critical values
result$critical_value_eps02
result$critical_value_eps05

# Rejection decisions
result$reject_eps02
result$reject_eps05

## ----message=FALSE------------------------------------------------------------
library(fracdiff)
set.seed(123)

# Simulate ARFIMA(0, d, 0) with d = 0.4
x_genuine <- fracdiff.sim(n = 1000, d = 0.4)$series

# Visualize
oldpar <- par(mfrow = c(1, 2))
plot(x_genuine, type = "l", main = "ARFIMA(0, 0.4, 0) Series", 
     ylab = "Value", col = "darkgreen", lwd = 0.8)
grid()

# ACF shows hyperbolic decay
acf(x_genuine, main = "ACF: Genuine Long Memory", 
    col = "darkgreen", lwd = 2)

par(oldpar)

## -----------------------------------------------------------------------------
result_genuine <- Longmemorytest(x_genuine, demean = TRUE, alpha = 0.05, 
                                  print_results = TRUE)

## -----------------------------------------------------------------------------
set.seed(456)
n <- 1000
shift_point <- 500

# White noise with level shift
x_spurious <- rnorm(n)
x_spurious[shift_point:n] <- x_spurious[shift_point:n] + 3

# Visualize
oldpar <- par(mfrow = c(1, 2))
plot(x_spurious, type = "l", main = "White Noise + Level Shift",
     ylab = "Value", col = "darkred", lwd = 0.8)
abline(v = shift_point, lty = 2, col = "blue", lwd = 2)
grid()

# ACF shows spurious persistence
acf(x_spurious, main = "ACF: Spurious Long Memory",
    col = "darkred", lwd = 2)

par(oldpar)

## -----------------------------------------------------------------------------
result_spurious <- Longmemorytest(x_spurious, demean = TRUE, alpha = 0.05)

## ----echo=FALSE---------------------------------------------------------------
cv_table <- data.frame(
  Alpha = c("10%", "5%", "2.5%", "1%"),
  `ε = 0.02` = c(1.118, 1.252, 1.374, 1.517),
  `ε = 0.05` = c(1.022, 1.155, 1.277, 1.426),
  check.names = FALSE
)
knitr::kable(cv_table, align = "c", 
             caption = "Asymptotic Critical Values")

## -----------------------------------------------------------------------------
sessionInfo()

