## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
 collapse = TRUE,
 comment = "#>"
)

## ----libraries, message = FALSE, warning = FALSE------------------------------
library(dplyr)
library(rbmi)
library(rbmiUtils)

## ----example-data-------------------------------------------------------------
set.seed(42)

dat <- data.frame(
 USUBJID = factor(rep(paste0("SUBJ-", 1:20), each = 4)),
 AVISIT = factor(
   rep(c("Week 4", "Week 8", "Week 12", "Week 16"), 20),
   levels = c("Week 4", "Week 8", "Week 12", "Week 16")
 ),
 TRT = factor(rep(c("Placebo", "Drug A"), each = 40)),
 BASE = rep(round(rnorm(20, 50, 10), 1), each = 4),
 STRATA = factor(rep(sample(c("Low", "High"), 20, replace = TRUE), each = 4))
)

# Generate CHG with some missing values
dat$CHG <- round(rnorm(80, mean = -2, sd = 3), 1)

# Create missing data patterns:
# - Subjects 3, 8: monotone dropout at Week 12
# - Subject 15: intermittent missing at Week 8
# - Subject 18: monotone dropout at Week 16
dat$CHG[dat$USUBJID == "SUBJ-3" & dat$AVISIT %in% c("Week 12", "Week 16")] <- NA
dat$CHG[dat$USUBJID == "SUBJ-8" & dat$AVISIT %in% c("Week 12", "Week 16")] <- NA
dat$CHG[dat$USUBJID == "SUBJ-15" & dat$AVISIT == "Week 8"] <- NA
dat$CHG[dat$USUBJID == "SUBJ-18" & dat$AVISIT == "Week 16"] <- NA

# Add discontinuation flag
dat$DISCFL <- ifelse(
 dat$USUBJID %in% c("SUBJ-3", "SUBJ-8") & dat$AVISIT == "Week 12",
 "Y",
 ifelse(
   dat$USUBJID == "SUBJ-18" & dat$AVISIT == "Week 16",
   "Y",
   "N"
 )
)

head(dat, 12)

## ----define-vars--------------------------------------------------------------
vars <- set_vars(
 subjid = "USUBJID",
 visit = "AVISIT",
 group = "TRT",
 outcome = "CHG",
 covariates = c("BASE", "STRATA"),
 strategy = "STRATEGY"
)

## ----validate-----------------------------------------------------------------
# This will pass validation
validate_data(dat, vars)

## ----validate-error, error = TRUE---------------------------------------------
try({
# Create problematic data
bad_dat <- dat
bad_dat$TRT <- as.character(bad_dat$TRT)  # Should be factor
bad_dat$BASE[1] <- NA  # Covariate with missing value

# This will report all issues at once
tryCatch(
 validate_data(bad_dat, vars),
 error = function(e) cat(e$message)
)
})

## ----summarise-missing--------------------------------------------------------
miss <- summarise_missingness(dat, vars)

## ----by-visit-----------------------------------------------------------------
print(miss$by_visit)

## ----patterns-----------------------------------------------------------------
print(miss$patterns)

## ----summary------------------------------------------------------------------
print(miss$summary)

## ----prepare-ice--------------------------------------------------------------
data_ice <- prepare_data_ice(
 data = dat,
 vars = vars,
 ice_col = "DISCFL",
 strategy = "JR"  # Jump to Reference
)

print(data_ice)

## ----workflow, eval = FALSE---------------------------------------------------
# library(rbmi)
# library(rbmiUtils)
# 
# # 1. Validate data
# validate_data(dat, vars)
# 
# # 2. Understand missing patterns
# miss <- summarise_missingness(dat, vars)
# print(miss$summary)
# 
# # 3. Prepare ICE data if needed
# data_ice <- prepare_data_ice(dat, vars, ice_col = "DISCFL", strategy = "JR")
# 
# # 4. Define method
# method <- method_bayes(
#  n_samples = 100,
#  control = control_bayes(warmup = 200, thin = 2)
# )
# 
# # 5. Run imputation
# draws_obj <- draws(
#  data = dat,
#  vars = vars,
#  data_ice = data_ice,
#  method = method
# )
# 
# # 6. Continue with impute() and analyse()

