% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AUCuniform_trap.R
\encoding{UTF-8}
\name{AUCuniform_trap}
\alias{AUCuniform_trap}
\title{Calculates the uniform \emph{AUC} and uniform \emph{Se*} by the
trapezoidal method.}
\usage{
AUCuniform_trap(
  mat,
  by = 0.1,
  deleteBins = NULL,
  w = NULL,
  plot = FALSE,
  plot.compare = FALSE,
  plot.adds = FALSE
)
}
\arguments{
\item{mat}{A matrix with two columns. The first column must contain the
suitability values (i.e., the classification rule); the second column must
contain the presences and absences.}

\item{by}{Size of the suitability intervals (i.e., bins). By default,
\code{by} = 0.1.}

\item{deleteBins}{A vector (e.g., from 1 to 10 if \code{by} = 0.1) with the
bins that have to be excluded (1 for [0,0.1), 10 for [0.9,1]) from the
calculation of the uniform statistics; \code{NULL} by default.}

\item{w}{A vector with the weights associated to each case. If \code{NULL}
(default), then the uniform \emph{AUC} (\emph{uAUC}) and uniform \emph{Se*}
(\emph{uSe*}) are calculated.}

\item{plot}{Logical. Indicates whether or not the observed ROC curve is
plotted (gray dots).}

\item{plot.compare}{Logical. Indicates whether or not the weighed ROC curve
is plotted (black line).}

\item{plot.adds}{Logical. Indicates whether or not the negative diagonal and
the points of equivalence (weighted and unweighted) are added to the ROC
plot.}
}
\value{
A list with the following elements:

\code{AUC}: the \emph{AUC} value (non-uniform, unweighted), a numeric
value between 0 and 1.

\code{Se}: the \emph{Se*} value (non-uniform, unweighted), a numeric
value between 0 and 1.

\code{bins}: a table with the sample size of each bin (only if
\code{w = NULL}).

\code{uAUC}: the uniform \emph{AUC} value (only if \code{w = NULL}).

\code{uSe}: the uniform \emph{Se*} value (only if \code{w = NULL}).

\code{wAUC}: the weighted \emph{AUC} estimated with the vector
\code{w}

\code{wSe}: the weighted \emph{Se*} estimated with the vector
\code{w}
}
\description{
This function computes the uniform \emph{AUC} (\emph{uAUC}) and
uniform \emph{Se*} (\emph{uSe*}) using the weighted trapezoidal method
instead of the weighted bootstrapping method used in \code{AUCuniform} and
originally proposed in Jiménez-Valverde (2022). This procedure reduces bias 
and improves the coverage of confidence intervals (Jiménez-Valverde 2024). 
Additionally, the weights vector associated to each case can be customized. 
See Jiménez-Valverde (2024) for details.
}
\details{
This function calculates the uniform \emph{AUC} (\emph{uAUC}) and
uniform \emph{Se*} (\emph{uSe*}) using the weighted trapezoidal method as
suggested in Jiménez-Valverde (2024). A warning message will be shown if
the sample size of any bin is zero. Another warning message will be shown if
the sample size of any bin is lower than 15. In such case, trimming should be
considered using \code{deleteBins} (Jiménez-Valverde 2022). Alternatively,
the weights associated to each case can be fully customized with the \code{w}
parameter (Jiménez-Valverde 2024). In this case, no warnings regarding
sample size issues are raised, and \code{deleteBins} is not used. The
\emph{AUC} (non-uniform, unweighted) is estimated non-parametrically by the
trapezoidal rule, which is equivalent to the Wilcoxon-based estimation 
(Hanley & McNeil 1982) used in \code{AUCuniform}. \emph{Se*} is calculated as 
in \code{AUCuniform}.
}
\examples{
\donttest{
suit<-rbeta(100, 2, 2) #Generate suitability values
random<-runif(100)
sp<-ifelse(random < suit, 1, 0) #Generate presence-absence data
result<-AUCuniform_trap(cbind(suit, sp), plot = TRUE, plot.compare = TRUE)
result$AUC #Get the AUC
result$uAUC #Get the uAUC. Note how it is closer to the reference value of
            #0.83 since the suitability values are simulated to be
            #well-calibrated (see Jimenez-Valverde 2022).
            }
}
\references{
Hanley, J. A. & McNeil, B. J. (1982). The Meaning and Use of the
    Area under a Receiver Operating Characteristic (ROC) Curve.
    \emph{Radiology}., 143, 29-36.

    Jiménez-Valverde, A. (2022). The uniform AUC: dealing with the
    representativeness effect in presence-absence models. \emph{Methods Ecol.
    Evol.}, 13, 1224-1236.

    Jiménez-Valverde, A. (2024). Improving the uniform AUC (uAUC): towards a 
    case-by-case weighting evaluation in species distribution models. 
    \emph{In preparation}.
}
\keyword{internal}
