/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 

#define H5HF_MAN_WRITE_CHECK_PLINE(HDR)                                                                      \
    {                                                                                                        \
        if (!((HDR)->checked_filters)) {                                                                     \
            if ((HDR)->pline.nused)                                                                          \
                if (H5Z_can_apply_direct(&((HDR)->pline)) < 0)                                               \
                    HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, FAIL, "I/O filters can't operate on this heap");     \
                                                                                                             \
            (HDR)->checked_filters = true;                                                                   \
        }                                                                                        \
    }

static herr_t H5HF__man_op_real(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data,
                                unsigned op_flags);

herr_t
H5HF__man_insert(H5HF_hdr_t *hdr, size_t obj_size, const void *obj, void *_id)
{
    H5HF_free_section_t *sec_node    = NULL;        
    H5HF_direct_t       *dblock      = NULL;        
    haddr_t              dblock_addr = HADDR_UNDEF; 
    size_t               dblock_size;               
    uint8_t             *id = (uint8_t *)_id;       
    size_t               blk_off;                   
    htri_t               node_found;                
    herr_t               ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(obj_size > 0);
    assert(obj);
    assert(id);

    
    H5HF_MAN_WRITE_CHECK_PLINE(hdr)

    
    if ((node_found = H5HF__space_find(hdr, (hsize_t)obj_size, &sec_node)) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't locate free space in fractal heap");

    
    if (!node_found)
        
        if (H5HF__man_dblock_new(hdr, obj_size, &sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL, "can't create fractal heap direct block");

    
    if (sec_node->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW ||
        sec_node->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW) {

        
        if (H5HF__man_iblock_alloc_row(hdr, &sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't break up row section");
    } 
    assert(sec_node->sect_info.type == H5HF_FSPACE_SECT_SINGLE);

    
    if (sec_node->sect_info.state == H5FS_SECT_SERIALIZED)
        if (H5HF__sect_single_revive(hdr, sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't revive single free section");
    assert(sec_node->sect_info.state == H5FS_SECT_LIVE);

    
    if (H5HF__sect_single_dblock_info(hdr, sec_node, &dblock_addr, &dblock_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve direct block information");

    
    if (NULL == (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, sec_node->u.single.parent,
                                                   sec_node->u.single.par_entry, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load fractal heap direct block");

    

    
    H5_CHECK_OVERFLOW((sec_node->sect_info.addr - dblock->block_off), hsize_t, size_t);
    blk_off = (size_t)(sec_node->sect_info.addr - dblock->block_off);

    
    assert(sec_node->sect_info.size >= obj_size);

    
    if (H5HF__sect_single_reduce(hdr, sec_node, obj_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce single section node");
    sec_node = NULL;

    
    {
        uint8_t *p; 

        
        p = dblock->blk + blk_off;

        
        H5MM_memcpy(p, obj, obj_size);
        p += obj_size;

        
        assert((size_t)(p - (dblock->blk + blk_off)) == obj_size);
    } 

    
    H5HF_MAN_ID_ENCODE(id, hdr, (dblock->block_off + blk_off), obj_size);

    
    hdr->man_nobjs++;

    
    if (H5HF__hdr_adj_free(hdr, -(ssize_t)obj_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't adjust free space for heap");

done:
    
    if (ret_value < 0)
        if (sec_node && H5HF__sect_single_free((H5FS_section_info_t *)sec_node) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to release section node");

    
    if (dblock && H5AC_unprotect(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock, H5AC__DIRTIED_FLAG) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap direct block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p)
{

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(id);
    assert(obj_len_p);

    
    id++;

    
    id += hdr->heap_off_size;

    
    UINT64DECODE_VAR(id, *obj_len_p, hdr->heap_len_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

void
H5HF__man_get_obj_off(const H5HF_hdr_t *hdr, const uint8_t *id, hsize_t *obj_off_p)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(id);
    assert(obj_off_p);

    
    id++;

    
    UINT64DECODE_VAR(id, *obj_off_p, hdr->heap_off_size);

    FUNC_LEAVE_NOAPI_VOID
} 

static herr_t
H5HF__man_op_real(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data, unsigned op_flags)
{
    H5HF_direct_t *dblock = NULL;       
    unsigned       dblock_access_flags; 
                                        
    haddr_t  dblock_addr = HADDR_UNDEF; 
    size_t   dblock_size;               
    unsigned dblock_cache_flags = 0;    
    hsize_t  obj_off;                   
    size_t   obj_len;                   
    size_t   blk_off;                   
    uint8_t *p;                         
    herr_t   ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(op);

    
    if (op_flags & H5HF_OP_MODIFY) {
        
        H5HF_MAN_WRITE_CHECK_PLINE(hdr)

        dblock_access_flags = H5AC__NO_FLAGS_SET;
        dblock_cache_flags  = H5AC__DIRTIED_FLAG;
    } 
    else {
        dblock_access_flags = H5AC__READ_ONLY_FLAG;
        dblock_cache_flags  = H5AC__NO_FLAGS_SET;
    } 

    
    id++;

    
    UINT64DECODE_VAR(id, obj_off, hdr->heap_off_size);
    UINT64DECODE_VAR(id, obj_len, hdr->heap_len_size);

    
    if (obj_off == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "invalid fractal heap offset");
    if (obj_off > hdr->man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object offset too large");
    if (obj_len == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "invalid fractal heap object size");
    if (obj_len > hdr->man_dtable.cparam.max_direct_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object size too large for direct block");
    if (obj_len > hdr->max_man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object should be standalone");

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        
        dblock_addr = hdr->man_dtable.table_addr;
        dblock_size = hdr->man_dtable.cparam.start_block_size;

        
        if (NULL ==
            (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, NULL, 0, dblock_access_flags)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap direct block");
    } 
    else {
        H5HF_indirect_t *iblock;      
        bool             did_protect; 
        unsigned         entry;       

        
        if (H5HF__man_dblock_locate(hdr, obj_off, &iblock, &entry, &did_protect, H5AC__READ_ONLY_FLAG) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of section");

        
        dblock_addr = iblock->ents[entry].addr;
        H5_CHECK_OVERFLOW((hdr->man_dtable.row_block_size[entry / hdr->man_dtable.cparam.width]), hsize_t,
                          size_t);
        dblock_size = (size_t)hdr->man_dtable.row_block_size[entry / hdr->man_dtable.cparam.width];

        
        if (!H5_addr_defined(dblock_addr)) {
            
            if (H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL,
                            "unable to release fractal heap indirect block");

            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap ID not in allocated direct block");
        } 

        
        if (NULL == (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, iblock, entry,
                                                       dblock_access_flags))) {
            
            if (H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL,
                            "unable to release fractal heap indirect block");

            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap direct block");
        } 

        
        if (H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
        iblock = NULL;
    } 

    
    assert((obj_off - dblock->block_off) < (hsize_t)dblock_size);
    blk_off = (size_t)(obj_off - dblock->block_off);

    
    if (blk_off < (size_t)H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr))
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "object located in prefix of direct block");

    
    if ((blk_off + obj_len) > dblock_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "object overruns end of direct block");

    
    p = dblock->blk + blk_off;

    
    if (op(p, obj_len, op_data) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "application's callback failed");

done:
    
    if (dblock && H5AC_unprotect(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock, dblock_cache_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap direct block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_read(H5HF_hdr_t *hdr, const uint8_t *id, void *obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(obj);

    
    if (H5HF__man_op_real(hdr, id, H5HF__op_read, obj, 0) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_write(H5HF_hdr_t *hdr, const uint8_t *id, const void *obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(obj);

    
    H5_WARN_CAST_AWAY_CONST_OFF
    if (H5HF__man_op_real(hdr, id, H5HF__op_write, (void *)obj, H5HF_OP_MODIFY) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");
    H5_WARN_CAST_AWAY_CONST_ON

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(op);

    
    if (H5HF__man_op_real(hdr, id, op, op_data, 0) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_remove(H5HF_hdr_t *hdr, const uint8_t *id)
{
    H5HF_free_section_t *sec_node    = NULL;  
    H5HF_indirect_t     *iblock      = NULL;  
    bool                 did_protect = false; 
    hsize_t              obj_off;             
    size_t               obj_len;             
    size_t               dblock_size;         
    hsize_t              dblock_block_off;    
    unsigned             dblock_entry;        
    size_t               blk_off;             
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);

    
    H5HF_MAN_WRITE_CHECK_PLINE(hdr)

    
    id++;

    
    UINT64DECODE_VAR(id, obj_off, hdr->heap_off_size);
    UINT64DECODE_VAR(id, obj_len, hdr->heap_len_size);

    
    if (obj_off == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "invalid fractal heap offset");
    if (obj_off > hdr->man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object offset too large");
    if (obj_len == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "invalid fractal heap object size");
    if (obj_len > hdr->man_dtable.cparam.max_direct_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object size too large for direct block");
    if (obj_len > hdr->max_man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap object should be standalone");

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        
        dblock_size      = hdr->man_dtable.cparam.start_block_size;
        dblock_block_off = 0;
        dblock_entry     = 0;
    } 
    else {
        
        if (H5HF__man_dblock_locate(hdr, obj_off, &iblock, &dblock_entry, &did_protect, H5AC__NO_FLAGS_SET) <
            0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of section");

        
        if (!H5_addr_defined(iblock->ents[dblock_entry].addr))
            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "fractal heap ID not in allocated direct block");

        
        H5_CHECK_OVERFLOW((hdr->man_dtable.row_block_size[dblock_entry / hdr->man_dtable.cparam.width]),
                          hsize_t, size_t);
        dblock_size = (size_t)(hdr->man_dtable.row_block_size[dblock_entry / hdr->man_dtable.cparam.width]);

        
        
        dblock_block_off = iblock->block_off;
        dblock_block_off += hdr->man_dtable.row_block_off[dblock_entry / hdr->man_dtable.cparam.width];
        dblock_block_off += hdr->man_dtable.row_block_size[dblock_entry / hdr->man_dtable.cparam.width] *
                            (dblock_entry % hdr->man_dtable.cparam.width);
    } 

    
    assert((obj_off - dblock_block_off) < (hsize_t)dblock_size);
    blk_off = (size_t)(obj_off - dblock_block_off);

    
    if (blk_off < (size_t)H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr))
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "object located in prefix of direct block");

    
    if ((blk_off + obj_len) > dblock_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "object overruns end of direct block");

    
    if (NULL == (sec_node = H5HF__sect_single_new(obj_off, obj_len, iblock, dblock_entry)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create section for direct block's free space");

    
    if (iblock) {
        if (H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
        iblock = NULL;
    } 

    
    if (H5HF__hdr_adj_free(hdr, (ssize_t)obj_len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't adjust free space for heap");

    
    hdr->man_nobjs--;

    
    if (H5HF__space_add(hdr, sec_node, H5FS_ADD_RETURNED_SPACE) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add direct block free space to global list");
    sec_node = NULL;

done:
    if (ret_value < 0) {
        
        if (sec_node && H5HF__sect_single_free((H5FS_section_info_t *)sec_node) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to release section node");
    } 

    
    if (iblock && H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");

    FUNC_LEAVE_NOAPI(ret_value)
} 
