/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"      
#include "H5Eprivate.h"     
#include "H5FDpkg.h"        
#include "H5FDonion_priv.h" 
#include "H5MMprivate.h"    

herr_t
H5FD__onion_ingest_history(H5FD_onion_history_t *history_out, H5FD_t *raw_file, haddr_t addr, haddr_t size)
{
    unsigned char *buf       = NULL;
    uint32_t       sum       = 0;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(history_out);
    assert(raw_file);

    
    history_out->record_locs = NULL;

    if (H5FD_get_eof(raw_file, H5FD_MEM_DRAW) < (addr + size))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "header indicates history beyond EOF");

    if (NULL == (buf = H5MM_malloc(sizeof(char) * size)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "can't allocate buffer space");

    if (H5FD_set_eoa(raw_file, H5FD_MEM_DRAW, (addr + size)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTSET, FAIL, "can't modify EOA");

    if (H5FD_read(raw_file, H5FD_MEM_DRAW, addr, size, buf) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "can't read history from file");

    if (H5FD__onion_history_decode(buf, history_out) != size)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDECODE, FAIL, "can't decode history (initial)");

    sum = H5_checksum_fletcher32(buf, size - 4);
    if (history_out->checksum != sum)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "checksum mismatch between buffer and stored");

    if (history_out->n_revisions > 0)
        if (NULL == (history_out->record_locs =
                         H5MM_calloc(history_out->n_revisions * sizeof(H5FD_onion_record_loc_t))))
            HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "can't allocate record pointer list");

    if (H5FD__onion_history_decode(buf, history_out) != size)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDECODE, FAIL, "can't decode history (final)");

done:
    H5MM_xfree(buf);
    if (ret_value < 0)
        H5MM_xfree(history_out->record_locs);

    FUNC_LEAVE_NOAPI(ret_value)
} 

uint64_t
H5FD__onion_write_history(H5FD_onion_history_t *history, H5FD_t *file, haddr_t off_start,
                          haddr_t filesize_curr)
{
    uint32_t       _sum      = 0; 
    uint64_t       size      = 0;
    unsigned char *buf       = NULL;
    uint64_t       ret_value = 0;

    FUNC_ENTER_PACKAGE

    if (NULL == (buf = H5MM_malloc(H5FD_ONION_ENCODED_SIZE_HISTORY +
                                   (H5FD_ONION_ENCODED_SIZE_RECORD_POINTER * history->n_revisions))))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, 0, "can't allocate buffer for updated history");

    if (0 == (size = H5FD__onion_history_encode(history, buf, &_sum)))
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, 0, "problem encoding updated history");

    if ((size + off_start > filesize_curr) && (H5FD_set_eoa(file, H5FD_MEM_DRAW, off_start + size) < 0))
        HGOTO_ERROR(H5E_VFL, H5E_CANTSET, 0, "can't modify EOA for updated history");

    if (H5FD_write(file, H5FD_MEM_DRAW, off_start, size, buf) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, 0, "can't write history as intended");

    ret_value = size;

done:
    H5MM_xfree(buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t H5_ATTR_NO_OPTIMIZE
H5FD__onion_history_decode(unsigned char *buf, H5FD_onion_history_t *history)
{
    uint32_t       ui32        = 0;
    uint32_t       sum         = 0;
    uint64_t       ui64        = 0;
    uint64_t       n_revisions = 0;
    uint8_t       *ui8p        = NULL;
    unsigned char *ptr         = NULL;
    size_t         ret_value   = 0;

    FUNC_ENTER_PACKAGE

    assert(buf != NULL);
    assert(history != NULL);
    assert(H5FD_ONION_HISTORY_VERSION_CURR == history->version);

    if (strncmp((const char *)buf, H5FD_ONION_HISTORY_SIGNATURE, 4))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid signature");

    if (H5FD_ONION_HISTORY_VERSION_CURR != buf[4])
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid version");

    ptr = buf + 8;

    H5MM_memcpy(&ui64, ptr, 8);
    ui8p = (uint8_t *)&ui64;
    UINT64DECODE(ui8p, n_revisions);
    ptr += 8;

    if (0 == history->n_revisions) {
        history->n_revisions = n_revisions;
        ptr += H5FD_ONION_ENCODED_SIZE_RECORD_POINTER * n_revisions;
    }
    else {
        if (history->n_revisions != n_revisions)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0,
                        "history argument suggests different revision count than encoded buffer");
        if (NULL == history->record_locs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "list is NULL -- cannot populate");

        for (uint64_t i = 0; i < n_revisions; i++) {
            H5FD_onion_record_loc_t *rloc = &history->record_locs[i];

            
            uint64_t record_size;
            uint64_t phys_addr;

            H5MM_memcpy(&ui64, ptr, 8);
            ui8p = (uint8_t *)&ui64;
            UINT64DECODE(ui8p, phys_addr);
            H5_CHECKED_ASSIGN(rloc->phys_addr, haddr_t, phys_addr, uint64_t);
            ptr += 8;

            H5MM_memcpy(&ui64, ptr, 8);
            ui8p = (uint8_t *)&ui64;
            UINT64DECODE(ui8p, record_size);
            H5_CHECKED_ASSIGN(rloc->record_size, hsize_t, record_size, uint64_t);
            ptr += 8;

            H5MM_memcpy(&ui32, ptr, 4);
            ui8p = (uint8_t *)&ui32;
            UINT32DECODE(ui8p, rloc->checksum);
            ptr += 4;
        }
    }

    sum = H5_checksum_fletcher32(buf, (size_t)(ptr - buf));

    H5MM_memcpy(&ui32, ptr, 4);
    ui8p = (uint8_t *)&ui32;
    UINT32DECODE(ui8p, history->checksum);
    ptr += 4;

    if (sum != history->checksum)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "checksum mismatch");

    ret_value = (size_t)(ptr - buf);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5FD__onion_history_encode(H5FD_onion_history_t *history, unsigned char *buf, uint32_t *checksum)
{
    unsigned char *ptr      = buf;
    size_t         vers_u32 = (uint32_t)history->version; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(history != NULL);
    assert(H5FD_ONION_HISTORY_VERSION_CURR == history->version);
    assert(buf != NULL);
    assert(checksum != NULL);

    H5MM_memcpy(ptr, H5FD_ONION_HISTORY_SIGNATURE, 4);
    ptr += 4;
    UINT32ENCODE(ptr, vers_u32);
    UINT64ENCODE(ptr, history->n_revisions);
    if (history->n_revisions > 0) {
        assert(history->record_locs != NULL);
        for (uint64_t i = 0; i < history->n_revisions; i++) {
            H5FD_onion_record_loc_t *rloc = &history->record_locs[i];

            
            uint64_t phys_addr;
            uint64_t record_size;

            H5_CHECKED_ASSIGN(phys_addr, uint64_t, rloc->phys_addr, haddr_t);
            H5_CHECKED_ASSIGN(record_size, uint64_t, rloc->record_size, hsize_t);

            UINT64ENCODE(ptr, phys_addr);
            UINT64ENCODE(ptr, record_size);
            UINT32ENCODE(ptr, rloc->checksum);
        }
    }
    *checksum = H5_checksum_fletcher32(buf, (size_t)(ptr - buf));
    UINT32ENCODE(ptr, *checksum);

    FUNC_LEAVE_NOAPI((size_t)(ptr - buf))
} 
