/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5A_FRIEND     
#include "H5Omodule.h" 
#define H5O_TESTING    

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5ACprivate.h" 
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Opkg.h"      

htri_t
H5O__is_attr_dense_test(hid_t oid)
{
    H5O_t      *oh = NULL;                    
    H5O_ainfo_t ainfo;                        
    H5O_loc_t  *loc;                          
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = FAIL;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        assert(H5O__msg_count_real(oh, H5O_MSG_ATTR) == 0);

        ret_value = true;
    } 
    else
        ret_value = false;

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5O__is_attr_empty_test(hid_t oid)
{
    H5O_t      *oh       = NULL;              
    H5B2_t     *bt2_name = NULL;              
    H5O_ainfo_t ainfo;                        
    htri_t      ainfo_exists = false;         
    H5O_loc_t  *loc;                          
    hsize_t     nattrs;                       
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = FAIL;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    if (oh->version > H5O_VERSION_1) {
        
        if ((ainfo_exists = H5A__get_ainfo(loc->file, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    nattrs = H5O__msg_count_real(oh, H5O_MSG_ATTR);

    
    if (oh->version > H5O_VERSION_1) {
        if (ainfo_exists) {
            
            if (H5_addr_defined(ainfo.fheap_addr)) {
                
                assert(nattrs == 0);

                
                H5_BEGIN_TAG(loc->addr)

                
                if (NULL == (bt2_name = H5B2_open(loc->file, ainfo.name_bt2_addr, NULL)))
                    HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTOPENOBJ, FAIL,
                                    "unable to open v2 B-tree for name index");

                
                H5_END_TAG

                
                if (H5B2_get_nrec(bt2_name, &nattrs) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTCOUNT, FAIL,
                                "unable to retrieve # of records from name index");
            } 

            
            assert(nattrs == ainfo.nattrs);
        } 
        else
            assert(nattrs == 0);
    } 

    
    ret_value = (nattrs == 0) ? true : false;

done:
    
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__num_attrs_test(hid_t oid, hsize_t *nattrs)
{
    H5O_t      *oh       = NULL;              
    H5B2_t     *bt2_name = NULL;              
    H5O_ainfo_t ainfo;                        
    H5O_loc_t  *loc;                          
    hsize_t     obj_nattrs;                   
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    obj_nattrs = H5O__msg_count_real(oh, H5O_MSG_ATTR);

    
    if (oh->version > H5O_VERSION_1) {
        
        if (H5_addr_defined(ainfo.fheap_addr)) {
            
            assert(obj_nattrs == 0);

            
            H5_BEGIN_TAG(loc->addr)

            
            if (NULL == (bt2_name = H5B2_open(loc->file, ainfo.name_bt2_addr, NULL)))
                HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

            
            H5_END_TAG

            
            if (H5B2_get_nrec(bt2_name, &obj_nattrs) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOUNT, FAIL, "unable to retrieve # of records from name index");
        } 

        
        assert(obj_nattrs == ainfo.nattrs);
    } 

    
    *nattrs = obj_nattrs;

done:
    
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_dense_info_test(hid_t oid, hsize_t *name_count, hsize_t *corder_count)
{
    H5O_t      *oh         = NULL;            
    H5B2_t     *bt2_name   = NULL;            
    H5B2_t     *bt2_corder = NULL;            
    H5O_ainfo_t ainfo;                        
    H5O_loc_t  *loc;                          
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    H5_BEGIN_TAG(loc->addr)

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR_TAG(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (!H5_addr_defined(ainfo.fheap_addr))
        HGOTO_DONE_TAG(FAIL);
    if (!H5_addr_defined(ainfo.name_bt2_addr))
        HGOTO_DONE_TAG(FAIL);

    
    if (NULL == (bt2_name = H5B2_open(loc->file, ainfo.name_bt2_addr, NULL)))
        HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    if (H5B2_get_nrec(bt2_name, name_count) < 0)
        HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTCOUNT, FAIL, "unable to retrieve # of records from name index");

    
    if (H5_addr_defined(ainfo.corder_bt2_addr)) {
        
        if (NULL == (bt2_corder = H5B2_open(loc->file, ainfo.corder_bt2_addr, NULL)))
            HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for creation order index");

        
        if (H5B2_get_nrec(bt2_corder, corder_count) < 0)
            HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTCOUNT, FAIL,
                            "unable to retrieve # of records from creation order index");
    } 
    else
        *corder_count = 0;

    
    H5_END_TAG

done:
    
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for creation order index");
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__check_msg_marked_test(hid_t oid, bool flag_val)
{
    H5O_t      *oh = NULL;           
    H5O_loc_t  *loc;                 
    H5O_mesg_t *idx_msg;             
    unsigned    idx;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    for (idx = 0, idx_msg = &oh->mesg[0]; idx < oh->nmesgs; idx++, idx_msg++)
        if (idx_msg->type->id == H5O_UNKNOWN_ID) {
            
            if (((idx_msg->flags & H5O_MSG_FLAG_WAS_UNKNOWN) > 0) != flag_val)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                            "'unknown' message has incorrect 'was unknown' flag value");

            
            break;
        } 

    
    if (idx == oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "'unknown' message type not found");

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__expunge_chunks_test(const H5O_loc_t *loc)
{
    H5O_t  *oh = NULL;           
    haddr_t chk_addr[16];        
    size_t  nchunks;             
    size_t  u;                   
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (oh = H5O_protect(loc, H5AC__NO_FLAGS_SET, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    nchunks = oh->nchunks;
    assert(0 < nchunks && nchunks < NELMTS(chk_addr));

    
    for (u = 0; u < oh->nchunks; u++)
        chk_addr[u] = oh->chunk[u].addr;

    
    if (H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header");

    
    
    for (u = nchunks - 1; u < nchunks; u--)
        if (H5AC_expunge_entry(loc->file, (u == 0 ? H5AC_OHDR : H5AC_OHDR_CHK), chk_addr[u],
                               H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTEXPUNGE, FAIL, "unable to expunge object header chunk");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__get_rc_test(const H5O_loc_t *loc, unsigned *rc)
{
    H5O_t *oh        = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(rc);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    *rc = oh->nlink;

done:
    
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_get_chunkno_test(hid_t oid, unsigned msg_type, unsigned *chunk_num)
{
    H5O_t      *oh = NULL;                    
    H5O_loc_t  *loc;                          
    H5O_mesg_t *idx_msg;                      
    unsigned    idx;                          
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    for (idx = 0, idx_msg = &oh->mesg[0]; idx < oh->nmesgs; idx++, idx_msg++)
        if (idx_msg->type->id == msg_type) {
            
            *chunk_num = idx_msg->chunkno;

            
            break;
        } 

    
    if (idx == oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "message of type not found");

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_move_to_new_chunk_test(hid_t oid, unsigned msg_type)
{
    H5O_t      *oh = NULL;                    
    H5O_loc_t  *loc;                          
    H5O_mesg_t *curr_msg;                     
    unsigned    idx;                          
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (loc = H5O_get_loc(oid)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object not found");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == (oh = H5O_protect(loc, H5AC__NO_FLAGS_SET, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    for (idx = 0, curr_msg = &oh->mesg[0]; idx < oh->nmesgs; idx++, curr_msg++)
        if (curr_msg->type->id == msg_type) {
            H5O_msg_alloc_info_t found_msg;                       
            unsigned             msg_chunkno = curr_msg->chunkno; 
            uint8_t             *end_chunk_data =
                (oh->chunk[msg_chunkno].image + oh->chunk[msg_chunkno].size) -
                (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[msg_chunkno].gap); 
            uint8_t *end_msg    = curr_msg->raw + curr_msg->raw_size;    
            size_t   gap_size   = 0; 
            size_t   null_size  = 0; 
            unsigned null_msgno = 0; 
            size_t   total_size;     
            size_t   new_idx;        

            
            if (end_msg == end_chunk_data)
                gap_size = oh->chunk[msg_chunkno].gap;
            else {
                H5O_mesg_t *tmp_msg; 
                unsigned    v;       

                
                for (v = 0, tmp_msg = &oh->mesg[0]; v < oh->nmesgs; v++, tmp_msg++) {
                    if (tmp_msg->type->id == H5O_NULL_ID &&
                        (tmp_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)) == end_msg) {
                        null_msgno = v;
                        null_size  = (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + tmp_msg->raw_size;
                        break;
                    } 

                    

                } 
            }     

            
            total_size = curr_msg->raw_size + gap_size + null_size;

            
            found_msg.msgno      = (int)idx;
            found_msg.id         = curr_msg->type->id;
            found_msg.chunkno    = msg_chunkno;
            found_msg.gap_size   = gap_size;
            found_msg.null_size  = null_size;
            found_msg.total_size = total_size;
            found_msg.null_msgno = null_msgno;

            
            
            if (H5O__alloc_chunk(loc->file, oh, (curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh)),
                                 oh->nmesgs, &found_msg, &new_idx) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't allocate new object header chunk");

            
            break;
        } 

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 
