/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5Pprivate.h"  
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

typedef struct {
    H5G_stat_t *statbuf;     
    bool        follow_link; 
    H5F_t      *loc_file;    
} H5G_trav_goi_t;

static herr_t H5G__get_objinfo_cb(H5G_loc_t *grp_loc , const char *name, const H5O_link_t *lnk,
                                  H5G_loc_t *obj_loc, void *_udata ,
                                  H5G_own_loc_t *own_loc );
#endif 

#ifndef H5_NO_DEPRECATED_SYMBOLS

H5G_obj_t
H5G_map_obj_type(H5O_type_t obj_type)
{
    H5G_obj_t ret_value = H5G_UNKNOWN; 

    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    switch (obj_type) {
        case H5O_TYPE_GROUP:
            ret_value = H5G_GROUP;
            break;

        case H5O_TYPE_DATASET:
            ret_value = H5G_DATASET;
            break;

        case H5O_TYPE_NAMED_DATATYPE:
            ret_value = H5G_TYPE;
            break;

        case H5O_TYPE_MAP:
            

        case H5O_TYPE_UNKNOWN:
        case H5O_TYPE_NTYPES:
        default:
            ret_value = H5G_UNKNOWN;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Gcreate1(hid_t loc_id, const char *name, size_t size_hint)
{
    void             *grp = NULL; 
    H5VL_object_t    *vol_obj;    
    H5VL_loc_params_t loc_params;
    hid_t             tmp_gcpl  = H5I_INVALID_HID; 
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no name given");
    if (size_hint > UINT32_MAX)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "size_hint cannot be larger than UINT32_MAX");

    
    if (size_hint > 0) {
        H5O_ginfo_t     ginfo;    
        H5P_genplist_t *gc_plist; 

        
        if (NULL == (gc_plist = (H5P_genplist_t *)H5I_object(H5P_GROUP_CREATE_DEFAULT)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property list");

        
        if ((tmp_gcpl = H5P_copy_plist(gc_plist, false)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "unable to copy the creation property list");

        
        if (NULL == (gc_plist = (H5P_genplist_t *)H5I_object(tmp_gcpl)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property list");

        
        if (H5P_get(gc_plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "can't get group info");

        
        H5_CHECKED_ASSIGN(ginfo.lheap_size_hint, uint32_t, size_hint, size_t);
        if (H5P_set(gc_plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTSET, H5I_INVALID_HID, "can't set group info");
    }
    else
        tmp_gcpl = H5P_GROUP_CREATE_DEFAULT;

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, H5I_INVALID_HID, "can't set collective metadata read info");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL ==
        (grp = H5VL_group_create(vol_obj, &loc_params, name, H5P_LINK_CREATE_DEFAULT, tmp_gcpl,
                                 H5P_GROUP_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5I_INVALID_HID, "unable to create group");

    
    if ((ret_value = H5VL_register(H5I_GROUP, grp, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register group");

done:
    if (H5I_INVALID_HID != tmp_gcpl && tmp_gcpl != H5P_GROUP_CREATE_DEFAULT)
        if (H5I_dec_ref(tmp_gcpl) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release property list");

    if (H5I_INVALID_HID == ret_value)
        if (grp && H5VL_group_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release group");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Gopen1(hid_t loc_id, const char *name)
{
    void             *grp     = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no name");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (grp = H5VL_group_open(vol_obj, &loc_params, name, H5P_GROUP_ACCESS_DEFAULT,
                                       H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open group");

    
    if ((ret_value = H5VL_register(H5I_GROUP, grp, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register group");

done:
    if (H5I_INVALID_HID == ret_value)
        if (grp && H5VL_group_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release group");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Glink(hid_t cur_loc_id, H5G_link_t type, const char *cur_name, const char *new_name)
{
    H5VL_link_create_args_t vol_cb_args;         
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!cur_name || !*cur_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no current name specified");
    if (!new_name || !*new_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no new name specified");

    
    if (H5CX_set_loc(cur_loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    if (type == H5L_TYPE_HARD) {
        H5VL_object_t    *vol_obj; 
        H5VL_loc_params_t new_loc_params;

        
        new_loc_params.type                         = H5VL_OBJECT_BY_NAME;
        new_loc_params.obj_type                     = H5I_get_type(cur_loc_id);
        new_loc_params.loc_data.loc_by_name.name    = new_name;
        new_loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

        
        if (NULL == (vol_obj = H5VL_vol_object(cur_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        vol_cb_args.op_type                                                = H5VL_LINK_CREATE_HARD;
        vol_cb_args.args.hard.curr_obj                                     = H5VL_OBJ_DATA(vol_obj);
        vol_cb_args.args.hard.curr_loc_params.type                         = H5VL_OBJECT_BY_NAME;
        vol_cb_args.args.hard.curr_loc_params.obj_type                     = H5I_get_type(cur_loc_id);
        vol_cb_args.args.hard.curr_loc_params.loc_data.loc_by_name.name    = cur_name;
        vol_cb_args.args.hard.curr_loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

        
        if (H5VL_link_create(&vol_cb_args, vol_obj, &new_loc_params, H5P_LINK_CREATE_DEFAULT,
                             H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create link");
    } 
    else if (type == H5L_TYPE_SOFT) {
        H5VL_object_t    *vol_obj; 
        H5VL_loc_params_t loc_params;

        
        loc_params.type                         = H5VL_OBJECT_BY_NAME;
        loc_params.loc_data.loc_by_name.name    = new_name;
        loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
        loc_params.obj_type                     = H5I_get_type(cur_loc_id);

        
        if (NULL == (vol_obj = H5VL_vol_object(cur_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        vol_cb_args.op_type          = H5VL_LINK_CREATE_SOFT;
        vol_cb_args.args.soft.target = cur_name;

        
        if (H5VL_link_create(&vol_cb_args, vol_obj, &loc_params, H5P_LINK_CREATE_DEFAULT,
                             H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create link");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Not a valid link type");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Glink2(hid_t cur_loc_id, const char *cur_name, H5G_link_t type, hid_t new_loc_id, const char *new_name)
{
    H5VL_link_create_args_t vol_cb_args;         
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!cur_name || !*cur_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no current name specified");
    if (!new_name || !*new_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no new name specified");

    
    if (H5CX_set_loc(cur_loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    if (type == H5L_TYPE_HARD) {
        H5VL_object_t    *vol_obj1; 
        H5VL_object_t    *vol_obj2; 
        H5VL_loc_params_t new_loc_params;

        
        new_loc_params.type                         = H5VL_OBJECT_BY_NAME;
        new_loc_params.obj_type                     = H5I_get_type(new_loc_id);
        new_loc_params.loc_data.loc_by_name.name    = new_name;
        new_loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

        
        if (NULL == (vol_obj1 = H5VL_vol_object(cur_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
        if (NULL == (vol_obj2 = H5VL_vol_object(new_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        vol_cb_args.op_type                                                = H5VL_LINK_CREATE_HARD;
        vol_cb_args.args.hard.curr_obj                                     = H5VL_OBJ_DATA(vol_obj1);
        vol_cb_args.args.hard.curr_loc_params.type                         = H5VL_OBJECT_BY_NAME;
        vol_cb_args.args.hard.curr_loc_params.obj_type                     = H5I_get_type(cur_loc_id);
        vol_cb_args.args.hard.curr_loc_params.loc_data.loc_by_name.name    = cur_name;
        vol_cb_args.args.hard.curr_loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

        
        if (H5VL_link_create(&vol_cb_args, vol_obj2, &new_loc_params, H5P_LINK_CREATE_DEFAULT,
                             H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create link");
    } 
    else if (type == H5L_TYPE_SOFT) {
        H5VL_object_t    *vol_obj; 
        H5VL_loc_params_t loc_params;

        
        if (new_loc_id == H5L_SAME_LOC)
            new_loc_id = cur_loc_id;

        
        loc_params.type                         = H5VL_OBJECT_BY_NAME;
        loc_params.loc_data.loc_by_name.name    = new_name;
        loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
        loc_params.obj_type                     = H5I_get_type(new_loc_id);

        
        if (NULL == (vol_obj = H5VL_vol_object(new_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        vol_cb_args.op_type          = H5VL_LINK_CREATE_SOFT;
        vol_cb_args.args.soft.target = cur_name;

        
        if (H5VL_link_create(&vol_cb_args, vol_obj, &loc_params, H5P_LINK_CREATE_DEFAULT,
                             H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create link");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a valid link type");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gmove(hid_t src_loc_id, const char *src_name, const char *dst_name)
{
    H5VL_object_t    *vol_obj; 
    H5VL_loc_params_t loc_params1;
    H5VL_loc_params_t loc_params2;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5CX_set_loc(src_loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    loc_params1.type                         = H5VL_OBJECT_BY_NAME;
    loc_params1.obj_type                     = H5I_get_type(src_loc_id);
    loc_params1.loc_data.loc_by_name.name    = src_name;
    loc_params1.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

    loc_params2.type                         = H5VL_OBJECT_BY_NAME;
    loc_params2.loc_data.loc_by_name.name    = dst_name;
    loc_params2.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

    
    if (NULL == (vol_obj = H5VL_vol_object(src_loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_link_move(vol_obj, &loc_params1, NULL, &loc_params2, H5P_LINK_CREATE_DEFAULT,
                       H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTMOVE, FAIL, "couldn't move link");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gmove2(hid_t src_loc_id, const char *src_name, hid_t dst_loc_id, const char *dst_name)
{
    H5VL_object_t    *vol_obj1 = NULL; 
    H5VL_loc_params_t loc_params1;
    H5VL_object_t    *vol_obj2 = NULL; 
    H5VL_loc_params_t loc_params2;
    H5I_type_t        src_id_type = H5I_BADID, dst_id_type = H5I_BADID;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!src_name || !*src_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no current name specified");
    if (!dst_name || !*dst_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no destination name specified");

    
    if (src_loc_id == H5L_SAME_LOC && dst_loc_id == H5L_SAME_LOC)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "current and destination should not both be H5L_SAME_LOC");

    
    if (src_loc_id == H5L_SAME_LOC)
        src_loc_id = dst_loc_id;
    else if (dst_loc_id == H5L_SAME_LOC)
        dst_loc_id = src_loc_id;

    src_id_type = H5I_get_type(src_loc_id);
    if (!(H5I_GROUP == src_id_type || H5I_FILE == src_id_type))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid group (or file) ID, src_loc_id");

    dst_id_type = H5I_get_type(dst_loc_id);
    if (!(H5I_GROUP == dst_id_type || H5I_FILE == dst_id_type))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid group (or file) ID, dst_loc_id");

    
    if (H5CX_set_loc(dst_loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    loc_params1.type                         = H5VL_OBJECT_BY_NAME;
    loc_params1.loc_data.loc_by_name.name    = src_name;
    loc_params1.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    loc_params1.obj_type                     = src_id_type;

    
    loc_params2.type                         = H5VL_OBJECT_BY_NAME;
    loc_params2.loc_data.loc_by_name.name    = dst_name;
    loc_params2.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    loc_params2.obj_type                     = dst_id_type;

    
    if (NULL == (vol_obj1 = H5VL_vol_object(src_loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    
    if (NULL == (vol_obj2 = H5VL_vol_object(dst_loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_link_move(vol_obj1, &loc_params1, vol_obj2, &loc_params2, H5P_LINK_CREATE_DEFAULT,
                       H5P_LINK_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTMOVE, FAIL, "unable to move link");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gunlink(hid_t loc_id, const char *name)
{
    H5VL_object_t            *vol_obj;     
    H5VL_link_specific_args_t vol_cb_args; 
    H5VL_loc_params_t         loc_params;
    herr_t                    ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.obj_type                     = H5I_get_type(loc_id);
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    vol_cb_args.op_type = H5VL_LINK_DELETE;

    
    if (H5VL_link_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "couldn't delete link");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_linkval(hid_t loc_id, const char *name, size_t size, char *buf )
{
    H5VL_object_t       *vol_obj;     
    H5VL_link_get_args_t vol_cb_args; 
    H5VL_loc_params_t    loc_params;
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.obj_type                     = H5I_get_type(loc_id);
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    vol_cb_args.op_type               = H5VL_LINK_GET_VAL;
    vol_cb_args.args.get_val.buf      = buf;
    vol_cb_args.args.get_val.buf_size = size;

    
    if (H5VL_link_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get link value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gset_comment(hid_t loc_id, const char *name, const char *comment)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    obj_opt_args.set_comment.comment = comment;
    vol_cb_args.op_type              = H5VL_NATIVE_OBJECT_SET_COMMENT;
    vol_cb_args.args                 = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "unable to set comment value");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Gget_comment(hid_t loc_id, const char *name, size_t bufsize, char *buf )
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    size_t                             comment_len = 0; 
    int                                ret_value;       

    FUNC_ENTER_API(-1)

    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, -1, "no name specified");
    if (bufsize > 0 && !buf)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, -1, "no buffer specified");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, -1, "can't set collective metadata read info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, -1, "invalid location identifier");

    
    obj_opt_args.get_comment.buf         = buf;
    obj_opt_args.get_comment.buf_size    = bufsize;
    obj_opt_args.get_comment.comment_len = &comment_len;
    vol_cb_args.op_type                  = H5VL_NATIVE_OBJECT_GET_COMMENT;
    vol_cb_args.args                     = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, -1, "unable to get comment value");

    
    ret_value = (int)comment_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Giterate(hid_t loc_id, const char *name, int *idx_p, H5G_iterate_t op, void *op_data)
{
    H5VL_object_t                    *vol_obj;      
    H5VL_optional_args_t              vol_cb_args;  
    H5VL_native_group_optional_args_t grp_opt_args; 
    hsize_t                           last_obj = 0; 
    herr_t                            ret_value;    

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");
    if (idx_p && *idx_p < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no operator specified");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid identifier");

    
    grp_opt_args.iterate_old.loc_params.type                         = H5VL_OBJECT_BY_NAME;
    grp_opt_args.iterate_old.loc_params.loc_data.loc_by_name.name    = name;
    grp_opt_args.iterate_old.loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    grp_opt_args.iterate_old.loc_params.obj_type                     = H5I_get_type(loc_id);
    grp_opt_args.iterate_old.idx                                     = (hsize_t)(idx_p == NULL ? 0 : *idx_p);
    grp_opt_args.iterate_old.last_obj                                = &last_obj;
    grp_opt_args.iterate_old.op                                      = op;
    grp_opt_args.iterate_old.op_data                                 = op_data;
    vol_cb_args.op_type                                              = H5VL_NATIVE_GROUP_ITERATE_OLD;
    vol_cb_args.args                                                 = &grp_opt_args;

    
    if ((ret_value = H5VL_group_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)) <
        0)
        HERROR(H5E_SYM, H5E_BADITER, "error iterating over group's links");

    
    if (idx_p)
        *idx_p = (int)last_obj;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_num_objs(hid_t loc_id, hsize_t *num_objs )
{
    H5VL_object_t        *vol_obj = NULL;      
    H5VL_group_get_args_t vol_cb_args;         
    H5I_type_t            id_type;             
    H5G_info_t            grp_info;            
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    id_type = H5I_get_type(loc_id);
    if (!(H5I_GROUP == id_type || H5I_FILE == id_type))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid group (or file) ID");
    if (!num_objs)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bad pointer to # of objects");

    
    vol_cb_args.op_type = H5VL_GROUP_GET_INFO;
    if (H5VL_setup_self_args(loc_id, &vol_obj, &vol_cb_args.args.get_info.loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set object access arguments");
    vol_cb_args.args.get_info.ginfo = &grp_info;

    
    if (H5VL_group_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get group info");

    
    *num_objs = grp_info.nlinks;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_objinfo(hid_t loc_id, const char *name, bool follow_link, H5G_stat_t *statbuf )
{
    H5VL_object_t                    *vol_obj = NULL;      
    H5VL_optional_args_t              vol_cb_args;         
    H5VL_native_group_optional_args_t grp_opt_args;        
    herr_t                            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    grp_opt_args.get_objinfo.loc_params.type                         = H5VL_OBJECT_BY_NAME;
    grp_opt_args.get_objinfo.loc_params.loc_data.loc_by_name.name    = name;
    grp_opt_args.get_objinfo.loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    grp_opt_args.get_objinfo.loc_params.obj_type                     = H5I_get_type(loc_id);
    grp_opt_args.get_objinfo.follow_link                             = follow_link;
    grp_opt_args.get_objinfo.statbuf                                 = statbuf;
    vol_cb_args.op_type                                              = H5VL_NATIVE_GROUP_GET_OBJINFO;
    vol_cb_args.args                                                 = &grp_opt_args;

    
    if (H5VL_group_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get info for object: '%s'", name);

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5G__get_objinfo_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char *name, const H5O_link_t *lnk,
                    H5G_loc_t *obj_loc, void *_udata , H5G_own_loc_t *own_loc )
{
    H5G_trav_goi_t *udata     = (H5G_trav_goi_t *)_udata; 
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    if (lnk == NULL && obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "'%s' doesn't exist", name);

    
    if (udata->statbuf) {
        H5G_stat_t *statbuf = udata->statbuf; 

        
        if (H5F_get_fileno((obj_loc ? obj_loc : grp_loc)->oloc->file, &statbuf->fileno[0]) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unable to read fileno");

        
        if (udata->follow_link || !lnk || (lnk->type == H5L_TYPE_HARD)) {
            H5O_info2_t       dm_info;  
            H5O_native_info_t nat_info; 
            haddr_t           obj_addr; 

            
            
            assert(obj_loc);
            if (H5O_get_info(obj_loc->oloc, &dm_info, H5O_INFO_BASIC | H5O_INFO_TIME) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get data model object info");
            if (H5O_get_native_info(obj_loc->oloc, &nat_info, H5O_INFO_HDR) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get native object info");

            
            statbuf->type = H5G_map_obj_type(dm_info.type);

            
            if (H5VL_native_token_to_addr(obj_loc->oloc->file, H5I_FILE, dm_info.token, &obj_addr) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTUNSERIALIZE, FAIL,
                            "can't deserialize object token into address");

            statbuf->objno[0] = (unsigned long)(obj_addr);
#if H5_SIZEOF_UINT64_T > H5_SIZEOF_LONG
            statbuf->objno[1] = (unsigned long)(obj_addr >> 8 * sizeof(long));
#else
            statbuf->objno[1] = 0;
#endif
            
            statbuf->nlink = dm_info.rc;

            
            statbuf->mtime = dm_info.ctime;

            
            statbuf->ohdr.size    = nat_info.hdr.space.total;
            statbuf->ohdr.free    = nat_info.hdr.space.free;
            statbuf->ohdr.nmesgs  = nat_info.hdr.nmesgs;
            statbuf->ohdr.nchunks = nat_info.hdr.nchunks;
        } 
    }     

done:
    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__get_objinfo(const H5G_loc_t *loc, const char *name, bool follow_link, H5G_stat_t *statbuf )
{
    H5G_trav_goi_t udata;               
    char          *obj_path = NULL;     
    const char    *last;                
    size_t         name_len;            
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name && *name);

    
    if (statbuf)
        memset(statbuf, 0, sizeof(H5G_stat_t));

    
    udata.statbuf     = statbuf;
    udata.follow_link = follow_link;
    udata.loc_file    = loc->oloc->file;

    
    if (H5G_traverse(loc, name,
                     (unsigned)(follow_link ? H5G_TARGET_NORMAL : (H5G_TARGET_SLINK | H5G_TARGET_UDLINK)),
                     H5G__get_objinfo_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_EXISTS, FAIL, "name doesn't exist");

    
    name_len = strlen(name);
    last     = name + (name_len - 1);
    while (name_len > 0) {
        
        if ('/' == *last || '.' == *last) {
            name_len--;
            last--;
        }
        else
            break;
    }
    if (name_len > 0) {
        if (NULL == (obj_path = H5MM_strdup(name)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, FAIL, "memory allocation failed for object path string");

        *(obj_path + name_len) = '\0';
    }

    
    if (obj_path && statbuf && follow_link == 0) {
        H5L_info2_t linfo; 

        
        if (H5L_get_info(loc, obj_path, &linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get link info");

        if (linfo.type != H5L_TYPE_HARD) {
            statbuf->linklen = linfo.u.val_size;
            if (linfo.type == H5L_TYPE_SOFT)
                statbuf->type = H5G_LINK;
            else {
                
                assert(linfo.type >= H5L_TYPE_UD_MIN && linfo.type <= H5L_TYPE_MAX);
                statbuf->type = H5G_UDLINK;
            }
        }
    }

done:
    H5MM_xfree(obj_path);

    FUNC_LEAVE_NOAPI(ret_value)
} 

ssize_t
H5Gget_objname_by_idx(hid_t loc_id, hsize_t idx, char *name , size_t size)
{
    H5VL_object_t       *vol_obj;     
    H5VL_link_get_args_t vol_cb_args; 
    H5VL_loc_params_t    loc_params;
    size_t               name_len = 0; 
    ssize_t              ret_value;    

    FUNC_ENTER_API(-1)

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, (-1), "can't set collective metadata read info");

    
    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = ".";
    loc_params.loc_data.loc_by_idx.idx_type = H5_INDEX_NAME;
    loc_params.loc_data.loc_by_idx.order    = H5_ITER_INC;
    loc_params.loc_data.loc_by_idx.n        = idx;
    loc_params.loc_data.loc_by_idx.lapl_id  = H5P_LINK_ACCESS_DEFAULT;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid location identifier");

    
    vol_cb_args.op_type                 = H5VL_LINK_GET_NAME;
    vol_cb_args.args.get_name.name_size = size;
    vol_cb_args.args.get_name.name      = name;
    vol_cb_args.args.get_name.name_len  = &name_len;

    
    if (H5VL_link_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, (-1), "can't get object name");

    
    ret_value = (ssize_t)name_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

H5G_obj_t
H5Gget_objtype_by_idx(hid_t loc_id, hsize_t idx)
{
    H5VL_object_t         *vol_obj;     
    H5VL_object_get_args_t vol_cb_args; 
    H5VL_loc_params_t      loc_params;
    H5O_info2_t            oinfo;     
    H5G_obj_t              ret_value; 

    FUNC_ENTER_API(H5G_UNKNOWN)

    
    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = ".";
    loc_params.loc_data.loc_by_idx.idx_type = H5_INDEX_NAME;
    loc_params.loc_data.loc_by_idx.order    = H5_ITER_INC;
    loc_params.loc_data.loc_by_idx.n        = idx;
    loc_params.loc_data.loc_by_idx.lapl_id  = H5P_LINK_ACCESS_DEFAULT;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5G_UNKNOWN, "invalid location identifier");

    
    vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
    vol_cb_args.args.get_info.oinfo  = &oinfo;
    vol_cb_args.args.get_info.fields = H5O_INFO_BASIC;

    
    if (H5VL_object_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, H5G_UNKNOWN, "can't get object info");

    
    if (H5G_UNKNOWN == (ret_value = H5G_map_obj_type(oinfo.type)))
        HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, H5G_UNKNOWN, "can't determine object type");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
