
#' Play a game of Four in a Row
#'
#' @param playerOne A function that takes the current board as input and returns
#'   the next move (1-7) for Player 1.
#' @param playerTwo Same for Player 2. Note that both functions see their pieces
#'   as "X" and opponent's pieces as "O".
#' @param verbose Logical value indicating whether or not to print the final
#'   board to the console (default is \code{TRUE})
#'
#' @details
#' The game is played on a \eqn{6 \times 7} grid and players alternate placing
#' markers in one of the \eqn{7} columns. The piece will "fall" to the lowest
#' unoccupied space in that column. The game ends when one player wins by
#' getting four pieces in a row (horizontally, vertically, or diagonally).
#'
#' Note that every player will see their markers as \code{X}s when it is their turn.
#'
#' @returns Returns \code{1} or \code{2} to indicate whether Player 1 or Player
#'   2 was the winner. Returns \code{0} in the case of a tie.
#'
#' @examples
#' play4inaRow(randomBot, randomBot)
#'
#' @importFrom methods is
#' @export
play4inaRow <- function(playerOne, playerTwo, verbose = TRUE){
  # checks
  stopifnot(is.function(playerOne))
  stopifnot(is.function(playerTwo))

  # set up game
  game <- matrix('.', nrow = 6, ncol = 7)

  # do turns
  over <- FALSE
  turn <- 0
  autowin <- FALSE
  sets <- getSetsof4()
  while(!over){
    turn <- turn + 1
    # switch X's and O's
    game <- invertPieces(game)

    # do turn
    if(turn %% 2 == 1){
      # playerOne's turn
      move <- tryCatch(playerOne(game), error = function(e) e)
      # check for automatic losses (error or invalid move)
      if(is(move,'error') || !(move %in% 1:7) || !(any(game[,move]=='.'))){
        over <- TRUE
        winner <- 2
        autowin <- TRUE
        if(is(move, 'error')){
          warning(paste0('playerOne encountered an error on turn ',turn))
        }else if(!(move %in% 1:7)){
          warning(paste0('playerOne made invalid move: value not in 1-7 (turn ',
                         turn,')'))
        }else if(!(any(game[,move]=='.'))){
          warning(paste0('playerOne made invalid move: column ',move,
                         ' is full (turn ',turn,')'))
        }
      }
    } else {
      # playerTwo's turn
      move <- tryCatch(playerTwo(game), error = function(e) e)
      # check for automatic losses (error or invalid move)
      if(is(move,'error') || !(move %in% 1:7) || !(any(game[,move]=='.'))){
        over <- TRUE
        winner <- 1
        autowin <- TRUE
        if(is(move, 'error')){
          warning(paste0('playerTwo encountered an error on turn ',turn))
        }else if(!(move %in% 1:7)){
          warning(paste0('playerTwo made invalid move: value not in 1-7 (turn ',
                         turn,')'))
        }else if(!(any(game[,move]=='.'))){
          warning(paste0('playerTwo made invalid move: column ',move,
                         ' is full (turn ',turn,')'))
        }
      }
    }

    if(!over){
      # add X
      row <- max(which(game[,move] == '.'))
      game[row, move] <- 'X'

      # check for winner
      # only check possible sets containing the most recently played piece
      ind <- 6*(move-1)+row
      sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
      winner <- apply(sx, 1, function(set){
        all(game[set] == 'X')
      })
      if(any(winner)){
        over <- TRUE
        winningSetID <- which(sets==ind, arr.ind = TRUE)[which(winner),1]
        if(turn %% 2 == 0){
          winner <- 2
          # switch X's and O's
          game <- invertPieces(game)
        }else{
          winner <- 1
        }
      }
      # check for tie
      if(!over & sum(game=='.') == 0){
        over <- TRUE
        winner <- 0
      }
    }
  }

  if(verbose){
    if(winner == 0 | autowin){
      # print game
      cat('   ')
      cat(paste(1:7, collapse='  '))
      cat('\n')
      cat(' _______________________\n')
      for(i in 1:nrow(game)){
        cat(' | ')
        cat(paste(game[i,], collapse='  '))
        cat(' |\n')
      }
      cat(' -----------------------\n')
    }else{
      # show final board, highlight winning 4
      game[,] <- paste0(' ',game[,],' ')
      game[sets[winningSetID,]] <- gsub(' ','*', game[sets[winningSetID,]])

      cat('   ')
      cat(paste(1:7, collapse='  '))
      cat('\n')
      cat(' _______________________\n')
      for(i in 1:nrow(game)){
        cat(' |')
        cat(paste(game[i,], collapse=''))
        cat('|\n')
      }
      cat(' -----------------------\n')
    }
  }
  return(winner)
}



