#' @title Import ConceptLattice from JSON
#' @description Reconstructs a ConceptLattice object from a JSON string.
#' @param json_str A JSON string generated by \code{to_json()}.
#' @return A \code{ConceptLattice} object.
#' @export
lattice_from_json <- function(json_str) {
    check_needed_pkg("jsonlite", "import from JSON")

    data <- jsonlite::fromJSON(json_str, simplifyVector = FALSE)

    if (data$type != "ConceptLattice") {
        stop("Invalid JSON: type must be 'ConceptLattice'")
    }

    # Reconstruct Extents
    # data$concepts is a list of concepts
    n_concepts <- length(data$concepts)

    # Initialize sparse matrices
    n_objects <- length(data$objects)
    n_attributes <- length(data$attributes)

    # We need to rebuild pr_extents and pr_intents
    # pr_extents: objects x concepts
    # pr_intents: attributes x concepts

    extents_i <- integer(0)
    extents_j <- integer(0)

    intents_i <- integer(0)
    intents_j <- integer(0)

    # Loop through concepts to build triplets
    for (k in seq_len(n_concepts)) {
        c_data <- data$concepts[[k]]

        # Extent
        if (length(c_data$extent) > 0) {
            extents_i <- c(extents_i, unlist(c_data$extent))
            extents_j <- c(extents_j, rep(k, length(c_data$extent)))
        }

        # Intent
        if (length(c_data$intent) > 0) {
            intents_i <- c(intents_i, unlist(c_data$intent))
            intents_j <- c(intents_j, rep(k, length(c_data$intent)))
        }
    }

    # Create matrices
    if (length(extents_i) > 0) {
        pr_extents <- Matrix::sparseMatrix(
            i = extents_i,
            j = extents_j,
            dims = c(n_objects, n_concepts),
            x = 1
        )
    } else {
        pr_extents <- Matrix::Matrix(
            0,
            nrow = n_objects,
            ncol = n_concepts,
            sparse = TRUE
        )
    }

    if (length(intents_i) > 0) {
        pr_intents <- Matrix::sparseMatrix(
            i = intents_i,
            j = intents_j,
            dims = c(n_attributes, n_concepts),
            x = 1
        )
    } else {
        pr_intents <- Matrix::Matrix(
            0,
            nrow = n_attributes,
            ncol = n_concepts,
            sparse = TRUE
        )
    }

    # Context I
    I <- NULL
    if (!is.null(data$I)) {
        i_I <- unlist(data$I$indices)
        j_I <- unlist(data$I$columns)
        if (length(i_I) > 0) {
            I <- Matrix::sparseMatrix(
                i = i_I,
                j = j_I,
                dims = c(n_objects, n_attributes),
                x = 1,
                dimnames = list(unlist(data$objects), unlist(data$attributes))
            )
        } else {
            I <- Matrix::Matrix(
                0,
                nrow = n_objects,
                ncol = n_attributes,
                sparse = TRUE
            )
            dimnames(I) <- list(unlist(data$objects), unlist(data$attributes))
        }
    }

    if (is.null(I)) {
        stop("JSON must contain 'I' (Context) field to reconstruct Lattice.")
    }

    objects <- unlist(data$objects)
    attributes <- unlist(data$attributes)

    # Instantiate ConceptLattice with required arguments
    cl <- ConceptLattice$new(
        extents = pr_extents,
        intents = pr_intents,
        objects = objects,
        attributes = attributes,
        I = I
    )

    # Hierarchy restoration if available
    if (!is.null(data$hierarchy)) {
        # Using internal method set_state if available for hierarchy
        if (exists("set_state", where = cl)) {
            cl$set_state(list(
                extents = pr_extents,
                intents = pr_intents,
                objects = objects,
                attributes = attributes,
                hierarchy = data$hierarchy
            ))
        }
    }

    return(cl)
}
