% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gev_weighted.R
\name{gev_weighted}
\alias{gev_weighted}
\title{Weighted GEV ML Inference with Adjustment for Missing Data}
\usage{
gev_weighted(data, scheme = 1, block_length, block, init = "quartiles", ...)
}
\arguments{
\item{data}{Either
\itemize{
\item a numeric vector containing a time series of raw data,
\item an object returned from \code{\link{block_maxima}}, a list with components
\code{maxima}, \code{notNA} and \code{n},
\item a data frame or named list containing the same information (variables
\code{maxima}, \code{notNA} and \code{n}) as an object returned from \code{\link{block_maxima}},
such as the data frame \code{\link{BrestSurgeMaxima}}.
}}

\item{scheme}{A numeric scalar. Pass \code{scheme = 1} for the first weighting
scheme, or \code{scheme = 2} for the second weighting scheme, in McVittie and
Murphy (2025). Any value other than \code{1} or \code{2} result in an unweighted fit,
that is, all weight are set to 1. See \strong{Details}.}

\item{block_length}{A numeric scalar. Used calculate the maxima of disjoint
blocks of \code{block_length} contiguous values in the vector \code{data}.
If \code{length(data)} is not an integer multiple of \code{block_length} then
the values at the end of \code{data} that do not constitute a complete block
of length \code{block_length} are discarded, without warning.}

\item{block}{A numeric vector with the same length as \code{data}. The value of
\code{block[i]} indicates the block into which \code{data[i]} falls. For example,
\code{block} could provide the year in which observation \code{i} was observed.}

\item{init}{Either a character scalar, one of \code{"quartiles"} or \code{"moments"},
or a numeric vector of length 3 giving initial estimates of the GEV
location, scale and shape parameters: \eqn{\mu}, \eqn{\sigma} and
\eqn{\xi}. If \code{init = "quartiles"} then initial estimates of \eqn{\mu} and
\eqn{\sigma} are based on sample quartiles of block maxima, ignoring the
underlying numbers of non-missing raw data, and a value of 0 for
\eqn{\xi}. If \code{init = "moments"} then instead we use the sample mean and
variance of these maxima and an initial value of 0.1 for \eqn{\xi}.}

\item{...}{Further arguments to be passed to \code{\link[stats:optim]{stats::optim}}.}
}
\value{
A list, returned from \code{\link[stats:optim]{stats::optim}} (the MLEs are in the
component \code{par}), with the additional components:
\itemize{
\item \code{loglik}: value of the maximised log-likelihood.
\item \code{vcov}: estimated variance-covariance matrix of the parameters.
\item \code{se}: estimated standard errors of the parameters.
\item \code{maxima}: the vector of block maxima used to fit the model.
\item \code{notNA}: the number of non-missing raw values on which the maxima in
\code{maxima} are based.
\item \code{n}: the maximal block length, that is, the largest number of values that
could have been observed in each of these blocks.
\item \code{weights}: the weights used in the fitting.
}

The call to \code{gev_mle} is provided in the attribute \code{"call"}.

The class of the returned object is
\code{c("evmissing", "weighted_mle", "list")}.

Objects inheriting from class \code{"evmissing"} have \code{coef}, \code{logLik}, \code{nobs},
\code{summary}, \code{vcov} and \code{confint} methods.  See \code{\link{evmissing_methods}}.
}
\description{
Fits a GEV distribution to block maxima using maximum likelihood estimation,
with the option to make an adjustment for the numbers of non-missing raw
values in each block using one of the two weighting schemes proposed in
McVittie and Murphy (2025).
}
\details{
Suppose that a full (no missing values) block will contain \eqn{n}
raw values. Let \eqn{n_i} be the number of non-missing values, and
\eqn{m_i} the observed block maximum, in block \eqn{i}.
The contribution of block maximum \eqn{m_i} to the GEV log-likelihood is
weighted (multiplied) by the weight \eqn{w_i}. The set of weights depends
on the weighting scheme chosen by \code{scheme} as follows.
\itemize{
\item If \code{scheme = 1} then \eqn{w_i = n_i / n}, for \eqn{i = 1, ..., n}.
\item If \code{scheme = 2} then \eqn{w_i = \hat{F}(m_i) ^ {n - n_i}}, for
\eqn{i = 1, ..., n}, where \eqn{\hat{F}} is the empirical distribution
function of \eqn{m_1, ..., m_n}.
}

For any other value of \code{scheme} all weights are set to 1, that is, an
unweighted fit is performed.

For each weighting scheme, the larger the number \eqn{n - n_i} of missing
values the smaller the weight.
See McVittie and Murphy (2025) for further details.
}
\examples{
## Simulate raw data from an exponential distribution

set.seed(13032025)
blocks <- 50
block_length <- 365
sdata <- sim_data(blocks = blocks, block_length = block_length)

# sdata$data_full have no missing values
# sdata$data_miss have had missing values created artificially

## Fits to full data: fit0, fit 1 and fit2 should give the same results

# Fit a GEV distribution to block maxima from the full data
fit0 <- gev_mle(sdata$data_full, block_length = sdata$block_length)
summary(fit0)

# Fit to the full data using weighting scheme 1
fit1 <- gev_weighted(sdata$data_full, scheme = 1,
                     block_length = sdata$block_length)
summary(fit1)

# Fit to the full data using weighting scheme 2
fit2 <- gev_weighted(sdata$data_full, scheme = 2,
                     block_length = sdata$block_length)
summary(fit2)

## Fits to the data with missings

#  Make adjustment for the numbers of non-missing values per block
fit0 <- gev_mle(sdata$data_miss, block_length = sdata$block_length)
summary(fit0)

# Make adjustment using weighting scheme 1
fit1 <- gev_weighted(sdata$data_miss, scheme = 1,
                     block_length = sdata$block_length)
summary(fit1)

# Make adjustment using weighting scheme 2
fit2 <- gev_weighted(sdata$data_miss, scheme = 2,
                     block_length = sdata$block_length)
summary(fit2)
}
\references{
McVittie, J. H. and Murphy, O. A. (2025) Weighted Parameter
Estimators of the Generalized Extreme Value Distribution in the Presence
of Missing Observations. \doi{10.48550/arXiv.2506.15964}
}
