% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lookup_recode.R
\name{lookup_recode}
\alias{lookup_recode}
\title{Lookup table switch handler}
\usage{
lookup_recode(
  src,
  type = c("species", "specimen", "inpatient_admission_method",
    "inpatient_discharge_destination", "ecds_destination_code", "manual"),
  .import = NULL
)
}
\arguments{
\item{src}{Character vector (or column) of values to recode. Coerced to character if needed.}

\item{type}{Character scalar specifying the lookup to use. One of:
\code{'species'}, \code{'specimen'}, \code{'inpatient_admission_method'},
\code{'inpatient_discharge_destination'}, \code{'ecds_destination_code'}, \code{'manual'}.}

\item{.import}{A two‑element list in the format \code{list(new, old)} used only
when \code{type = 'manual'}. Each element must be a vector of equal length.}
}
\value{
A character vector containing the recoded values, aligned 1:1 with \code{src}.
Values not present in the lookup are returned unchanged.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

A function to recode values via named lookup tables (i.e call an epidm lookup table and
recode where we are aware of a new value). It routes to a specific lookup based on type,
returning a character vector where each input value has been mapped to its corresponding
replacement. If a value is not found in the lookup then the original value is returned.

Built‑in lookups include:
\itemize{
\item \strong{\code{species}}: Uses the \code{respeciate_organism} dataset to standardise and
reclassify organism names (e.g., historic → current nomenclature). This
supports consistent reporting across SGSS and other laboratory datasets.
\item \strong{\code{specimen}}: Uses the \code{specimen_type_grouping} dataset to assign raw
laboratory specimen types into harmonised specimen groups. This enables
consistent grouping for reporting, aggregation, and filtering.
\item \strong{\code{genus_gram_stain}}: Uses the \code{genus_gram_stain} lookup table, which
provides Gram stain classifications by bacterial genus. This reference is
manually maintained against the UKHSA SGSS database and supports rapid
filtering and high‑level organism categorisation. Users should raise an
issue or submit a pull request to the \code{epidm} GitHub repository if
an organism/genus is missing.
\item \strong{\code{lab_data}}: Uses the \code{lab_data} lookup dataset for harmonising
laboratory code systems and internal SGSS mappings, supporting standardised
laboratory result interpretation within surveillance pipelines.
\item \strong{\code{inpatient_admission_method}}: Uses the internal lookup table
\code{epidm:::group_inpatient_admission_method} to categorise raw hospital
admission method codes into operationally meaningful groups.
\item \strong{\code{inpatient_discharge_destination}}: Uses the internal table
\code{epidm:::group_inpatient_discharge_destination} to group hospital discharge
destination codes into standardised categories for inpatient pathway
analysis.
\item \strong{\code{ecds_destination_code}}: Uses the internal table
\code{epidm:::group_ecds_discharge_destination}, providing grouped mappings for
ECDS (Emergency Care Data Set) discharge codes.
\item \strong{\code{manual}}: Allows the user to supply their own lookup through
\code{.import = list(new, old)}. This is useful when working with local,
provisional, or evolving code sets not yet included in the package’s
centralised lookup tables.
}
}
\examples{
df <- data.frame(
  spec = c(
    sample(grep(")",
                respeciate_organism$previous_organism_name,
                value=TRUE,
                invert = TRUE),
           9),
    "ESCHERICHIA COLI","SARS-COV-2","CANDIDA AUREUS"),
  type = sample(specimen_type_grouping$specimen_type,12),
  date = sample(seq.Date(from = Sys.Date()-365,
                         to = Sys.Date(),
                         by = "day"),12)
)
df <- df[order(df$date),]

# show the data before the changes
df

# check the lookup tables
# observe the changes
head(respeciate_organism[1:2])
df$species <- lookup_recode(df$spec,'species')
df[,c('spec','species','date')]

head(specimen_type_grouping)
df$grp <- lookup_recode(df$type,'specimen')
df[,c('species','type','grp','date')]

# for a tidyverse use
# df \%>\% mutate(spec=lookup_recode(spec,'species))

# manual input of your own lookup
# .import=list(new,old)
lookup_recode(
  "ALCALIGENES DENITRIFICANS",
  type = 'manual',
  .import=list(respeciate_organism$organism_species_name,
               respeciate_organism$previous_organism_name)
  )

}
