% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pc.R
\name{pc}
\alias{pc}
\title{PC Algorithm for Causal Discovery}
\usage{
pc(engine = c("tetrad", "pcalg", "bnlearn"), test, alpha = 0.05, ...)
}
\arguments{
\item{engine}{Character; which engine to use. Must be one of:
\describe{
\item{\code{"tetrad"}}{\pkg{Tetrad} Java library.}
\item{\code{"pcalg"}}{\pkg{pcalg} R package.}
\item{\code{"bnlearn"}}{\pkg{bnlearn} R package.}
}}

\item{test}{Character; name of the conditional‐independence test.}

\item{alpha}{Numeric; significance level for the CI tests.}

\item{...}{Additional arguments passed to the chosen engine (e.g. test or algorithm parameters).}
}
\description{
Run the PC (Peter-Clark) algorithm for causal discovery using one of several engines.
}
\details{
For specific details on the supported tests and parameters for each engine, see:
\itemize{
\item \link{TetradSearch} for \pkg{Tetrad},
\item \link{PcalgSearch} for \pkg{pcalg},
\item \link{BnlearnSearch} for \pkg{bnlearn}.
}
}
\section{Recommendation}{

While it is possible to call the function returned directly with a data frame,
we recommend using \code{\link[=disco]{disco()}}. This provides a consistent interface and handles knowledge
integration.
}

\section{Value}{

A function that takes a single argument \code{data} (a data frame). When called,
this function returns a list containing:
\itemize{
\item \code{knowledge} A \code{Knowledge} object with the background knowledge
used in the causal discovery algorithm. See \code{\link[=knowledge]{knowledge()}} for how to construct it.
\item \code{caugi} A \code{\link[caugi:caugi]{caugi::caugi}} object (of class \code{PDAG}) representing the learned causal graph
from the causal discovery algorithm.
}
}

\examples{
data(tpc_example)

#### Using pcalg engine ####
# Recommended path using disco()
pc_pcalg <- pc(engine = "pcalg", test = "fisher_z", alpha = 0.05)
disco(tpc_example, pc_pcalg)

# or using pc_pcalg directly
pc_pcalg(tpc_example)

# With all algorithm arguments specified
pc_pcalg <- pc(
  engine = "pcalg",
  test = "fisher_z",
  alpha = 0.05,
  fixedGaps = NULL,
  fixedEdges = NULL,
  NAdelete = FALSE,
  m.max = 10,
  u2pd = "relaxed",
  skel.method = "original",
  conservative = TRUE,
  maj.rule = FALSE,
  solve.confl = TRUE,
  numCores = 1,
  verbose = FALSE
)

#### Using bnlearn engine with required knowledge ####
kn <- knowledge(
  tpc_example,
  starts_with("child") \%-->\% starts_with("youth")
)


# Recommended path using disco()
pc_bnlearn <- pc(engine = "bnlearn", test = "fisher_z", alpha = 0.05)
disco(tpc_example, pc_bnlearn, knowledge = kn)

# or using pc_bnlearn directly
pc_bnlearn <- pc_bnlearn |> set_knowledge(kn)
pc_bnlearn(tpc_example)


# With all algorithm arguments specified
pc_bnlearn <- pc(
  engine = "bnlearn",
  test = "fisher_z",
  alpha = 0.05,
  max.sx = 2,
  debug = FALSE,
  undirected = TRUE
)

disco(tpc_example, pc_bnlearn)

#### Using tetrad engine with tier knowledge ####
# Requires Tetrad to be installed
if (verify_tetrad()$installed && verify_tetrad()$java_ok) {
  kn <- knowledge(
    tpc_example,
    tier(
      child ~ tidyselect::starts_with("child"),
      youth ~ tidyselect::starts_with("youth"),
      oldage ~ tidyselect::starts_with("oldage")
    )
  )

  # Recommended path using disco()
  pc_tetrad <- pc(engine = "tetrad", test = "fisher_z", alpha = 0.05)
  disco(tpc_example, pc_tetrad, knowledge = kn)

  # or using pc_tetrad directly
  pc_tetrad <- pc_tetrad |> set_knowledge(kn)
  pc_tetrad(tpc_example)
}

# With all algorithm arguments specified
if (verify_tetrad()$installed && verify_tetrad()$java_ok) {
  pc_tetrad <- pc(
    engine = "tetrad",
    test = "fisher_z",
    alpha = 0.05,
    conflict_rule = 2,
    depth = 10,
    stable_fas = FALSE,
    guarantee_cpdag = TRUE
  )
  disco(tpc_example, pc_tetrad)
}
}
\seealso{
Other causal discovery algorithms: 
\code{\link{boss}()},
\code{\link{boss_fci}()},
\code{\link{fci}()},
\code{\link{ges}()},
\code{\link{gfci}()},
\code{\link{grasp}()},
\code{\link{grasp_fci}()},
\code{\link{gs}()},
\code{\link{iamb-family}},
\code{\link{sp_fci}()},
\code{\link{tfci}()},
\code{\link{tges}()},
\code{\link{tpc}()}
}
\concept{causal discovery algorithms}
\concept{cd_algorithms}
