# ============================================================================ #
# Diagnostic plots for betaregscale objects
#
# Provides 6 standard diagnostic plots following the betareg tradition:
#   1. Residuals vs indices
#   2. Cook's distance
#   3. Residuals vs linear predictor
#   4. Residuals vs fitted values
#   5. Half-normal plot with simulated envelope
#   6. Predicted vs observed
#
# Both base R and ggplot2 backends are available (gg = TRUE).
# ============================================================================ #

#' Diagnostic plots for beta interval regression
#'
#' @description
#' Produces up to six diagnostic plots for a fitted
#' \code{"brs"} model: residuals vs indices, Cook's
#' distance, residuals vs linear predictor, residuals vs fitted
#' values, a half-normal plot with simulated envelope, and
#' predicted vs observed.
#'
#' @param x      A fitted \code{"brs"} object.
#' @param which  Integer vector selecting which plots to draw
#'   (default \code{1:4}).
#' @param type   Character: residual type passed to
#'   \code{\link{residuals.brs}} (default \code{"rqr"}).
#' @param nsim   Integer: number of simulations for the half-normal
#'   envelope (default 100).
#' @param level  Numeric: confidence level for the envelope
#'   (default 0.9).
#' @param caption Character vector of panel captions.
#' @param sub.caption Subtitle; defaults to the model call.
#' @param ask    Logical: prompt before each page of plots?
#' @param gg     Logical: use ggplot2? (default \code{FALSE}).
#' @param title Optional global title for ggplot output. If \code{NULL},
#'   panel captions are used.
#' @param theme Optional ggplot2 theme object (e.g., \code{ggplot2::theme_bw()}).
#'   If \code{NULL}, a minimal theme is used.
#' @param ...    Further arguments passed to base \code{plot()}.
#'
#' @return Invisibly returns \code{x}.
#'
#' @seealso \code{\link{brs}}, \code{\link{residuals.brs}},
#'   \code{\link{autoplot.brs}}
#'
#' @examples
#' \donttest{
#' dat <- data.frame(
#'   y = c(
#'     0, 5, 20, 50, 75, 90, 100, 30, 60, 45,
#'     10, 40, 55, 70, 85, 25, 35, 65, 80, 15
#'   ),
#'   x1 = rep(c(1, 2), 10)
#' )
#' prep <- brs_prep(dat, ncuts = 100)
#' fit <- brs(y ~ x1, data = prep)
#' plot(fit, which = 1:4)
#' }
#'
#' @method plot brs
#' @importFrom stats qnorm fitted residuals hatvalues qqnorm quantile median
#' @importFrom graphics plot abline par mtext segments lines
#' @importFrom grDevices dev.interactive devAskNewPage adjustcolor
#' @export
plot.brs <- function(x,
                     which = 1:4,
                     type = "rqr",
                     nsim = 100L,
                     level = 0.9,
                     caption = c(
                       "Residuals vs indices",
                       "Cook's distance",
                       "Residuals vs linear predictor",
                       "Residuals vs fitted values",
                       "Half-normal plot",
                       "Predicted vs observed"
                     ),
                     sub.caption = NULL,
                     ask = prod(par("mfcol")) < length(which) &&
                       dev.interactive(),
                     gg = FALSE,
                     title = NULL,
                     theme = NULL,
                     ...) {
  .check_class(x)
  if (is.null(sub.caption)) {
    sub.caption <- deparse(x$call, width.cutoff = 80L)
    if (length(sub.caption) > 1L) sub.caption <- paste(sub.caption, collapse = " ")
  }

  if (gg) {
    .plot_gg(x,
      which = which, type = type, nsim = nsim, level = level,
      caption = caption, sub.caption = sub.caption,
      title = title, theme = theme
    )
  } else {
    .plot_base(x,
      which = which, type = type, nsim = nsim, level = level,
      caption = caption, sub.caption = sub.caption, ask = ask, ...
    )
  }

  invisible(x)
}


# -- Base R plotting -------------------------------------------------------- #

.plot_base <- function(x, which, type, nsim, level, caption, sub.caption,
                       ask, ...) {
  r <- residuals(x, type = type)
  mu_hat <- fitted(x, type = "mu")
  eta <- stats::make.link(x$link)$linkfun(mu_hat)
  n <- length(r)
  idx <- seq_len(n)

  # Cook's distance approximation: h_i * r_i^2 / (p * (1-h_i)^2)
  p <- x$npar
  V <- vcov(x)
  X <- x$model_matrices$X
  # Leverage via hat matrix: H = X (X'X)^{-1} X'
  h <- tryCatch(
    {
      XtXinv <- solve(crossprod(X))
      rowSums((X %*% XtXinv) * X)
    },
    error = function(e) rep(1 / n, n)
  )
  cooks <- (r^2 * h) / (p * (1 - h)^2)

  show <- which
  nplots <- length(show)

  if (nplots > 1L) {
    ncol <- min(nplots, 2L)
    nrow <- ceiling(nplots / ncol)
    op <- par(mfrow = c(nrow, ncol), oma = c(0, 0, 2, 0))
    on.exit(par(op))
  }

  if (ask) {
    oask <- devAskNewPage(TRUE)
    on.exit(devAskNewPage(oask), add = TRUE)
  }

  # 1. Residuals vs indices
  if (1L %in% show) {
    plot(idx, r,
      xlab = "Index", ylab = "Residuals",
      main = caption[1L], pch = 20, col = "gray40", ...
    )
    abline(h = 0, lty = 2, col = "red")
  }

  # 2. Cook's distance
  if (2L %in% show) {
    plot(idx, cooks,
      type = "h", xlab = "Index",
      ylab = "Cook's distance", main = caption[2L],
      col = "gray40", ...
    )
    abline(h = 4 / n, lty = 2, col = "red")
  }

  # 3. Residuals vs linear predictor
  if (3L %in% show) {
    plot(eta, r,
      xlab = "Linear predictor", ylab = "Residuals",
      main = caption[3L], pch = 20, col = "gray40", ...
    )
    abline(h = 0, lty = 2, col = "red")
  }

  # 4. Residuals vs fitted
  if (4L %in% show) {
    plot(mu_hat, r,
      xlab = "Fitted values", ylab = "Residuals",
      main = caption[4L], pch = 20, col = "gray40", ...
    )
    abline(h = 0, lty = 2, col = "red")
  }

  # 5. Half-normal plot with envelope
  if (5L %in% show) {
    .plot_halfnormal_base(x, r,
      nsim = nsim, level = level,
      caption = caption[5L], ...
    )
  }

  # 6. Predicted vs observed
  if (6L %in% show) {
    y_obs <- x$Y[, "yt"]
    plot(mu_hat, y_obs,
      xlab = "Predicted", ylab = "Observed",
      main = caption[6L], pch = 20, col = "gray40", ...
    )
    abline(0, 1, lty = 2, col = "red")
  }

  if (nplots > 1L) mtext(sub.caption, outer = TRUE, cex = 0.8)
}


.plot_halfnormal_base <- function(x, r, nsim, level, caption, ...) {
  n <- length(r)
  ar <- sort(abs(r))
  qth <- stats::qnorm((seq_len(n) + n - 0.125) / (2 * n + 0.5))

  # Simulated envelope
  env <- matrix(NA_real_, nrow = n, ncol = nsim)
  for (j in seq_len(nsim)) {
    env[, j] <- sort(abs(stats::rnorm(n)))
  }
  alpha_half <- (1 - level) / 2
  lo <- apply(env, 1, quantile, probs = alpha_half)
  hi <- apply(env, 1, quantile, probs = 1 - alpha_half)
  me <- apply(env, 1, median)

  yl <- range(c(ar, lo, hi))
  plot(qth, ar,
    xlab = "Half-normal quantiles",
    ylab = "|Residuals|", main = caption,
    ylim = yl, pch = 20, col = "gray40", ...
  )
  lines(qth, me, lty = 2, col = "gray60")
  lines(qth, lo, lty = 3, col = "red")
  lines(qth, hi, lty = 3, col = "red")
}


# -- ggplot2 plotting ------------------------------------------------------- #

.plot_gg <- function(x, which, type, nsim, level, caption, sub.caption, title, theme) {
  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop(
      "Package 'ggplot2' is required for gg = TRUE. ",
      "Install it with install.packages('ggplot2').",
      call. = FALSE
    )
  }

  theme_obj <- .resolve_gg_theme(theme)
  has_global_title <- !is.null(title) && nzchar(title)

  r <- residuals(x, type = type)
  mu_hat <- fitted(x, type = "mu")
  eta <- stats::make.link(x$link)$linkfun(mu_hat)
  n <- length(r)
  idx <- seq_len(n)
  y_obs <- x$Y[, "yt"]

  # Cook's distance
  p <- x$npar
  X <- x$model_matrices$X
  h <- tryCatch(
    {
      XtXinv <- solve(crossprod(X))
      rowSums((X %*% XtXinv) * X)
    },
    error = function(e) rep(1 / n, n)
  )
  cooks <- (r^2 * h) / (p * (1 - h)^2)

  df <- data.frame(
    idx = idx, r = r, mu = mu_hat, eta = eta,
    cooks = cooks, y_obs = y_obs
  )

  plots <- list()
  panel_title <- function(i) {
    if (has_global_title && length(which) == 1L) title else caption[i]
  }
  panel_subtitle <- function() {
    if (length(which) == 1L && !is.null(sub.caption) && nzchar(sub.caption)) sub.caption else NULL
  }

  if (1L %in% which) {
    plots[[length(plots) + 1L]] <- ggplot2::ggplot(df, ggplot2::aes(x = .data$idx, y = .data$r)) +
      ggplot2::geom_point(color = "gray40", size = 1) +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
      ggplot2::labs(title = panel_title(1L), subtitle = panel_subtitle(), x = "Index", y = "Residuals") +
      theme_obj
  }

  if (2L %in% which) {
    plots[[length(plots) + 1L]] <- ggplot2::ggplot(df, ggplot2::aes(x = .data$idx, y = .data$cooks)) +
      ggplot2::geom_segment(ggplot2::aes(xend = .data$idx, yend = 0),
        color = "gray40"
      ) +
      ggplot2::geom_hline(yintercept = 4 / n, linetype = "dashed", color = "red") +
      ggplot2::labs(title = panel_title(2L), subtitle = panel_subtitle(), x = "Index", y = "Cook's distance") +
      theme_obj
  }

  if (3L %in% which) {
    plots[[length(plots) + 1L]] <- ggplot2::ggplot(df, ggplot2::aes(x = .data$eta, y = .data$r)) +
      ggplot2::geom_point(color = "gray40", size = 1) +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
      ggplot2::labs(title = panel_title(3L), subtitle = panel_subtitle(), x = "Linear predictor", y = "Residuals") +
      theme_obj
  }

  if (4L %in% which) {
    plots[[length(plots) + 1L]] <- ggplot2::ggplot(df, ggplot2::aes(x = .data$mu, y = .data$r)) +
      ggplot2::geom_point(color = "gray40", size = 1) +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
      ggplot2::labs(title = panel_title(4L), subtitle = panel_subtitle(), x = "Fitted values", y = "Residuals") +
      theme_obj
  }

  if (5L %in% which) {
    plots[[length(plots) + 1L]] <- .plot_halfnormal_gg(
      r, nsim, level, panel_title(5L), panel_subtitle(), theme_obj
    )
  }

  if (6L %in% which) {
    plots[[length(plots) + 1L]] <- ggplot2::ggplot(df, ggplot2::aes(x = .data$mu, y = .data$y_obs)) +
      ggplot2::geom_point(color = "gray40", size = 1) +
      ggplot2::geom_abline(intercept = 0, slope = 1, linetype = "dashed", color = "red") +
      ggplot2::labs(title = panel_title(6L), subtitle = panel_subtitle(), x = "Predicted", y = "Observed") +
      theme_obj
  }

  # Arrange plots in a grid
  np <- length(plots)
  if (np == 1L) {
    print(plots[[1L]])
  } else {
    ncol <- min(np, 2L)
    nrow <- ceiling(np / ncol)
    # Use gridExtra if available, otherwise print sequentially
    if (requireNamespace("gridExtra", quietly = TRUE)) {
      gridExtra::grid.arrange(
        grobs = plots, ncol = ncol, nrow = nrow,
        top = if (has_global_title) title else NULL
      )
    } else {
      for (p in plots) print(p)
    }
  }
}


.plot_halfnormal_gg <- function(r, nsim, level, caption, subtitle = NULL, theme_obj = ggplot2::theme_minimal()) {
  n <- length(r)
  ar <- sort(abs(r))
  qth <- stats::qnorm((seq_len(n) + n - 0.125) / (2 * n + 0.5))

  env <- matrix(NA_real_, nrow = n, ncol = nsim)
  for (j in seq_len(nsim)) {
    env[, j] <- sort(abs(stats::rnorm(n)))
  }
  alpha_half <- (1 - level) / 2
  lo <- apply(env, 1, quantile, probs = alpha_half)
  hi <- apply(env, 1, quantile, probs = 1 - alpha_half)
  me <- apply(env, 1, median)

  df_hn <- data.frame(qth = qth, ar = ar, lo = lo, hi = hi, me = me)

  ggplot2::ggplot(df_hn, ggplot2::aes(x = .data$qth, y = .data$ar)) +
    ggplot2::geom_point(color = "gray40", size = 1) +
    ggplot2::geom_line(ggplot2::aes(y = .data$me),
      linetype = "dashed",
      color = "gray60"
    ) +
    ggplot2::geom_line(ggplot2::aes(y = .data$lo),
      linetype = "dotted",
      color = "red"
    ) +
    ggplot2::geom_line(ggplot2::aes(y = .data$hi),
      linetype = "dotted",
      color = "red"
    ) +
    ggplot2::labs(
      title = caption, subtitle = subtitle, x = "Half-normal quantiles",
      y = "|Residuals|"
    ) +
    theme_obj
}

.resolve_gg_theme <- function(theme) {
  if (is.null(theme)) {
    return(ggplot2::theme_minimal())
  }
  if (is.function(theme)) {
    return(theme())
  }
  theme
}
