## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  message = FALSE,
  warning = FALSE
)

## ----load-packages------------------------------------------------------------
library(afcharts)
library(ggplot2)
library(dplyr)
library(ggtext)
library(scales)

# Use gapminder data for cookbook charts
library(gapminder)

## ----line-charts-1, eval = FALSE----------------------------------------------
# gapminder |>
#   filter(country == "United Kingdom") |>
#   ggplot(aes(x = year, y = lifeExp)) +
#   geom_line(linewidth = 1, colour = af_colour_values["dark-blue"]) +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 5)) +
#   labs(
#     x = "Year",
#     y = NULL
#   )

## ----echo = FALSE-------------------------------------------------------------
gapminder |>
  filter(country == "United Kingdom") |>
  ggplot(aes(x = year, y = lifeExp)) +
  geom_line(linewidth = 1, colour = af_colour_values["dark-blue"]) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 5)) +
  labs(
    x = "Year",
    y = NULL
  )

## ----line-charts-2, eval = FALSE----------------------------------------------
# gapminder |>
#   filter(country %in% c("United Kingdom", "China")) |>
#   ggplot(
#     aes(
#       x = year, y = lifeExp,
#       colour = factor(country, levels = c("United Kingdom", "China"))
#     )
#   ) +
#   geom_line(linewidth = 1) +
#   theme_af(legend = "right") +
#   scale_colour_discrete_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 10)) +
#   labs(
#     x = "Year",
#     y = NULL,
#     colour = NULL
#   )

## ----fig.height = 5, echo = FALSE---------------------------------------------
gapminder |>
  filter(country %in% c("United Kingdom", "China")) |>
  ggplot(
    aes(
      x = year, y = lifeExp,
      colour = factor(country, levels = c("United Kingdom", "China"))
    )
  ) +
  geom_line(linewidth = 1) +
  theme_af(legend = "right") +
  scale_colour_discrete_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 10)) +
  labs(
    x = "Year",
    y = NULL,
    colour = NULL
  )

## ----bar-data-----------------------------------------------------------------
pop_bar_data <- gapminder |>
  filter(year == 2007 & continent == "Americas") |>
  slice_max(order_by = pop, n = 5)

## ----bar-charts-1, eval = FALSE-----------------------------------------------
# ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6),
#     expand = expansion(mult = c(0, 0.1)),
#   ) +
#   labs(
#     x = NULL,
#     y = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1)),
  ) +
  labs(
    x = NULL,
    y = NULL,
  )

## ----bar-charts-2, eval = FALSE-----------------------------------------------
# ggplot(pop_bar_data, aes(x = pop, y = reorder(country, pop))) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   theme_af(grid = "x", axis = "y", axis_title = "none") +
#   scale_x_continuous(
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6),
#     expand = expansion(mult = c(0, 0.1))
#   )

## ----echo = FALSE-------------------------------------------------------------
ggplot(pop_bar_data, aes(x = pop, y = reorder(country, pop))) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af(grid = "x", axis = "y", axis_title = "none") +
  scale_x_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1))
  )

## ----grouped-bar-chart, eval=FALSE--------------------------------------------
# grouped_bar_data <-
#   gapminder |>
#   filter(
#     year %in% c(1967, 2007) &
#       country %in% c("United Kingdom", "Ireland", "France", "Belgium")
#   )
# 
# ggplot(
#   grouped_bar_data,
#   aes(x = country, y = lifeExp, fill = as.factor(year))
# ) +
#   geom_bar(stat = "identity", position = "dodge") +
#   scale_y_continuous(
#     limits = c(0, 100),
#     breaks = c(seq(0, 100, 20)),
#     labels = c(seq(0, 100, 20)),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   theme_af(legend = "bottom") +
#   scale_fill_discrete_af() +
#   labs(
#     x = "Country",
#     y = NULL,
#     fill = NULL
#   )

## ----fig.height = 5.5, echo=FALSE---------------------------------------------
grouped_bar_data <-
  gapminder |>
  filter(
    year %in% c(1967, 2007) &
      country %in% c("United Kingdom", "Ireland", "France", "Belgium")
  )

ggplot(
  grouped_bar_data,
  aes(x = country, y = lifeExp, fill = as.factor(year))
) +
  geom_bar(stat = "identity", position = "dodge") +
  scale_y_continuous(
    limits = c(0, 100),
    breaks = c(seq(0, 100, 20)),
    labels = c(seq(0, 100, 20)),
    expand = expansion(mult = c(0, 0.1))
  ) +
  theme_af(legend = "bottom") +
  scale_fill_discrete_af() +
  labs(
    x = "Country",
    y = NULL,
    fill = NULL
  )

## ----stacked-bar-chart, eval = FALSE------------------------------------------
# stacked_bar_data <-
#   gapminder |>
#   filter(year == 2007) |>
#   mutate(
#     lifeExpGrouped = cut(
#       lifeExp,
#       breaks = c(0, 75, Inf),
#       labels = c("Under 75", "75+")
#     )
#   ) |>
#   group_by(continent, lifeExpGrouped) |>
#   summarise(n_countries = n(), .groups = "drop")
# 
# ggplot(
#   stacked_bar_data,
#   aes(x = continent, y = n_countries, fill = lifeExpGrouped)
# ) +
#   geom_bar(stat = "identity", position = "fill") +
#   theme_af(legend = "right") +
#   scale_y_continuous(
#     labels = scales::percent,
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   coord_cartesian(clip = "off") +
#   scale_fill_discrete_af() +
#   labs(
#     x = NULL,
#     y = NULL,
#     fill = "Life Expectancy",
#   )

## ----fig.height = 5.5, echo = FALSE-------------------------------------------
stacked_bar_data <-
  gapminder |>
  filter(year == 2007) |>
  mutate(
    lifeExpGrouped = cut(
      lifeExp,
      breaks = c(0, 75, Inf),
      labels = c("Under 75", "75+")
    )
  ) |>
  group_by(continent, lifeExpGrouped) |>
  summarise(n_countries = n(), .groups = "drop")

ggplot(
  stacked_bar_data,
  aes(x = continent, y = n_countries, fill = lifeExpGrouped)
) +
  geom_bar(stat = "identity", position = "fill") +
  theme_af(legend = "right") +
  scale_y_continuous(
    labels = scales::percent,
    expand = expansion(mult = c(0, 0.1))
  ) +
  coord_cartesian(clip = "off") +
  scale_fill_discrete_af() +
  labs(
    x = NULL,
    y = NULL,
    fill = "Life Expectancy",
  )

## ----histogram, eval = FALSE--------------------------------------------------
# gapminder |>
#   filter(year == 2007) |>
#   ggplot(aes(x = lifeExp)) +
#   geom_histogram(
#     binwidth = 5,
#     colour = "white",
#     fill = af_colour_values["dark-blue"]
#   ) +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 35),
#     breaks = c(seq(0, 35, 5)),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   labs(
#     x = NULL,
#     y = "Number of \ncountries",
#   )

## ----echo = FALSE-------------------------------------------------------------
gapminder |>
  filter(year == 2007) |>
  ggplot(aes(x = lifeExp)) +
  geom_histogram(
    binwidth = 5,
    colour = "white",
    fill = af_colour_values["dark-blue"]
  ) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 35),
    breaks = c(seq(0, 35, 5)),
    expand = expansion(mult = c(0, 0.1))
  ) +
  labs(
    x = NULL,
    y = "Number of \ncountries",
  )

## ----scatterplot, eval = FALSE------------------------------------------------
# gapminder |>
#   filter(year == 2007) |>
#   ggplot(aes(x = gdpPercap, y = lifeExp)) +
#   geom_point(colour = af_colour_values["dark-blue"]) +
#   theme_af(axis = "none", grid = "xy") +
#   scale_x_continuous(labels = scales::label_comma()) +
#   labs(
#     x = "GDP (US$, inflation-adjusted)",
#     y = "Life\nExpectancy\n(years)",
#   )

## ----fig.height = 5, echo = FALSE---------------------------------------------
gapminder |>
  filter(year == 2007) |>
  ggplot(aes(x = gdpPercap, y = lifeExp)) +
  geom_point(colour = af_colour_values["dark-blue"]) +
  theme_af(axis = "none", grid = "xy") +
  scale_x_continuous(labels = scales::label_comma()) +
  labs(
    x = "GDP (US$, inflation-adjusted)",
    y = "Life\nExpectancy\n(years)",
  )

## ----small-multiples, eval = FALSE--------------------------------------------
# gapminder |>
#   filter(continent != "Oceania") |>
#   group_by(continent, year) |>
#   summarise(pop = sum(as.numeric(pop)), .groups = "drop") |>
#   ggplot(aes(x = year, y = pop, fill = continent)) +
#   geom_area() +
#   theme_af(axis = "none", ticks = "none", legend = "none") +
#   scale_fill_discrete_af() +
#   facet_wrap(~ continent, ncol = 2, scales = "free_x") +
#   scale_x_continuous(
#     breaks = c(1952, 2007),
#     labels = c("1952", "2007"),
#     limits = c(1950, 2010)
#   ) +
#   scale_y_continuous(
#     breaks = c(0, 2e9, 4e9),
#     labels = c(0, "2bn", "4bn")
#   ) +
#   coord_cartesian(clip = "off") +
#   labs(
#     x = NULL,
#     y = NULL
#   )

## ----fig.height = 5.5, echo = FALSE-------------------------------------------
gapminder |>
  filter(continent != "Oceania") |>
  group_by(continent, year) |>
  summarise(pop = sum(as.numeric(pop)), .groups = "drop") |>
  ggplot(aes(x = year, y = pop, fill = continent)) +
  geom_area() +
  theme_af(axis = "none", ticks = "none", legend = "none") +
  scale_fill_discrete_af() +
  facet_wrap(~ continent, ncol = 2, scales = "free_x") +
  scale_x_continuous(
    breaks = c(1952, 2007),
    labels = c("1952", "2007"),
    limits = c(1950, 2010)
  ) +
  scale_y_continuous(
    breaks = c(0, 2e9, 4e9),
    labels = c(0, "2bn", "4bn")
  ) +
  coord_cartesian(clip = "off") +
  labs(
    x = NULL,
    y = NULL
  )

## ----pie-chart, eval = FALSE--------------------------------------------------
# stacked_bar_data |>
#   filter(continent == "Europe") |>
#   ggplot(aes(x = "", y = n_countries, fill = lifeExpGrouped)) +
#   geom_col(colour = "white", position = "fill") +
#   coord_polar(theta = "y") +
#   theme_af(grid = "none", axis = "none", ticks = "none") +
#   theme(axis.text = element_blank()) +
#   scale_fill_discrete_af() +
#   labs(
#     x = NULL,
#     y = NULL,
#     fill = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
stacked_bar_data |>
  filter(continent == "Europe") |>
  ggplot(aes(x = "", y = n_countries, fill = lifeExpGrouped)) +
  geom_col(colour = "white", position = "fill") +
  coord_polar(theta = "y") +
  theme_af(grid = "none", axis = "none", ticks = "none") +
  theme(axis.text = element_blank()) +
  scale_fill_discrete_af() +
  labs(
    x = NULL,
    y = NULL,
    fill = NULL,
  )

## ----focus-chart, eval = FALSE------------------------------------------------
# pop_bar_data |>
#   ggplot(aes(x = reorder(country, -pop), y = pop, fill = country == "Brazil")) +
#   geom_col() +
#   theme_af(legend = "none") +
#   scale_y_continuous(
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_fill_discrete_af("focus", reverse = TRUE) +
#   labs(
#     x = NULL,
#     y = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
pop_bar_data |>
  ggplot(aes(x = reorder(country, -pop), y = pop, fill = country == "Brazil")) +
  geom_col() +
  theme_af(legend = "none") +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_fill_discrete_af("focus", reverse = TRUE) +
  labs(
    x = NULL,
    y = NULL,
  )

## ----interactive-charts, eval = FALSE-----------------------------------------
# p <-
#   pop_bar_data |>
#   # Format text for tooltips
#   mutate(
#     tooltip = paste0(
#       "Country: ", country, "\n",
#       "Population (millions): ", round(pop / 10 ^ 6, 1)
#     )
#   ) |>
#   ggplot(aes(x = reorder(country, -pop), y = pop, text = tooltip)) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   theme_af(ticks = "x") +
#   theme(text = element_text(family = "")) +
#   scale_y_continuous(
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   labs(
#     x = NULL,
#     y = NULL
#   )
# 
# plotly::ggplotly(p, tooltip = "text") |>
#   plotly::config(
#     modeBarButtons = list(list("resetViews")),
#     displaylogo = FALSE
#   )

## ----echo = FALSE-------------------------------------------------------------
p <-
  pop_bar_data |>
  # Format text for tooltips
  mutate(
    tooltip = paste0(
      "Country: ", country, "\n",
      "Population (millions): ", round(pop / 10 ^ 6, 1)
    )
  ) |>
  ggplot(aes(x = reorder(country, -pop), y = pop, text = tooltip)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af(ticks = "x") +
  theme(text = element_text(family = "")) +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1))
  ) +
  labs(
    x = NULL,
    y = NULL
  )

plotly::ggplotly(p, tooltip = "text") |>
  plotly::config(
    modeBarButtons = list(list("resetViews")),
    displaylogo = FALSE
  )

## ----annotations-data---------------------------------------------------------
ann_data <- gapminder |>
  filter(country %in% c("United Kingdom", "China")) |>
  mutate(country = factor(country, levels = c("United Kingdom", "China")))

## ----annotations-1, eval = FALSE----------------------------------------------
# ann_data |>
#   ggplot(aes(x = year, y = lifeExp)) +
#   geom_line(
#     aes(colour = country),
#     linewidth = 1
#   ) +
#   theme_af(legend = "none") +
#   scale_colour_discrete_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_x_continuous(
#     limits = c(1952, 2017),
#     breaks = seq(1952, 2017, 10)
#   ) +
#   annotate(
#     geom = "label",
#     x = 2008, y = 73,
#     label = "China",
#     linewidth = NA,
#     hjust = 0,
#     vjust = 0.5
#   ) +
#   annotate(
#     geom = "label",
#     x = 2008,
#     y = 79.4,
#     label = "United Kingdom",
#     linewidth = NA,
#     hjust = 0,
#     vjust = 0.5
#   ) +
#   labs(
#     x = "Year",
#     y = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
ann_data |>
  ggplot(aes(x = year, y = lifeExp)) +
  geom_line(
    aes(colour = country),
    linewidth = 1
  ) +
  theme_af(legend = "none") +
  scale_colour_discrete_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(
    limits = c(1952, 2017),
    breaks = seq(1952, 2017, 10)
  ) +
  annotate(
    geom = "label",
    x = 2008, y = 73,
    label = "China",
    linewidth = NA,
    hjust = 0,
    vjust = 0.5
  ) +
  annotate(
    geom = "label",
    x = 2008,
    y = 79.4,
    label = "United Kingdom",
    linewidth = NA,
    hjust = 0,
    vjust = 0.5
  ) +
  labs(
    x = "Year",
    y = NULL,
  )

## ----annotations-2, eval = FALSE----------------------------------------------
# ann_labs <- ann_data |>
#   group_by(country) |>
#   mutate(min_year = min(year)) |>
#   filter(year == max(year)) |>
#   ungroup()
# 
# ann_data |>
#   ggplot(aes(x = year, y = lifeExp)) +
#   geom_line(
#     aes(colour = country),
#     linewidth = 1
#   ) +
#   theme_af(legend = "none") +
#   scale_colour_discrete_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_x_continuous(
#     limits = c(1952, 2017),
#     breaks = seq(1952, 2017, 10)
#   ) +
#   geom_label(
#     data = ann_labs,
#     aes(x = year, y = lifeExp, label = country),
#     hjust = 0,
#     vjust = 0.5,
#     nudge_x = 0.5,
#     linewidth = NA
#   ) +
#   labs(
#     x = "Year",
#     y = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
ann_labs <- ann_data |>
  group_by(country) |>
  mutate(min_year = min(year)) |>
  filter(year == max(year)) |>
  ungroup()

ann_data |>
  ggplot(aes(x = year, y = lifeExp)) +
  geom_line(
    aes(colour = country),
    linewidth = 1
  ) +
  theme_af(legend = "none") +
  scale_colour_discrete_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(
    limits = c(1952, 2017),
    breaks = seq(1952, 2017, 10)
  ) +
  geom_label(
    data = ann_labs,
    aes(x = year, y = lifeExp, label = country),
    hjust = 0,
    vjust = 0.5,
    nudge_x = 0.5,
    linewidth = NA
  ) +
  labs(
    x = "Year",
    y = NULL,
  )

## ----annotations-3, eval = FALSE----------------------------------------------
# ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   geom_text(
#     aes(label = round(pop / 1E6, 1)),
#     vjust = 1.2,
#     colour = "white"
#   ) +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   labs(
#     x = NULL,
#     y = NULL
#   )

## ----echo = FALSE-------------------------------------------------------------
ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  geom_text(
    aes(label = round(pop / 1E6, 1)),
    vjust = 1.2,
    colour = "white"
  ) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1))
  ) +
  labs(
    x = NULL,
    y = NULL
  )

## ----sorted, eval = FALSE-----------------------------------------------------
# population_chart <- pop_bar_data |>
#   ggplot(aes(x = pop, y = reorder(country, pop))) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   theme_af(axis = "y", grid = "x") +
#   labs(
#     x = NULL,
#     y = NULL
#   )
# print(population_chart)

## ----echo = FALSE-------------------------------------------------------------
population_chart <- pop_bar_data |>
  ggplot(aes(x = pop, y = reorder(country, pop))) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af(axis = "y", grid = "x") +
  labs(
    x = NULL,
    y = NULL
  )
print(population_chart)

## ----expand, eval = FALSE-----------------------------------------------------
# last_plot() +
#   scale_x_continuous(expand = expansion(mult = c(0, 0.1)))

## ----echo = FALSE-------------------------------------------------------------
last_plot() +
  scale_x_continuous(expand = expansion(mult = c(0, 0.1)))

## ----axis-limits-breaks-labels-custom, eval = FALSE---------------------------
# last_plot() +
#   scale_x_continuous(
#     breaks = seq(0, 400E6, 100E6),
#     labels = seq(0, 400, 100),
#     limits = c(0, 420E6),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   labs(
#     x = "Population (millions)"
#   )

## ----echo = FALSE-------------------------------------------------------------
last_plot() +
  scale_x_continuous(
    breaks = seq(0, 400E6, 100E6),
    labels = seq(0, 400, 100),
    limits = c(0, 420E6),
    expand = expansion(mult = c(0, 0.1))
  ) +
  labs(
    x = "Population (millions)"
  )

## ----axis-limits-breaks-labels-fct, eval = FALSE------------------------------
# 
# breaks_pretty <- pretty(pop_bar_data$pop)
# limits_pretty <- range(breaks_pretty)
# 
# last_plot() +
#   scale_x_continuous(
#     breaks = breaks_pretty,
#     labels = label_number(scale = 1E-6),
#     limits = limits_pretty,
#     expand = expansion(mult = c(0, 0.2))
#   ) +
#   labs(
#     x = "Population (millions)"
#   )
# 

## ----echo = FALSE-------------------------------------------------------------

breaks_pretty <- pretty(pop_bar_data$pop)
limits_pretty <- range(breaks_pretty)

last_plot() +
  scale_x_continuous(
    breaks = breaks_pretty,
    labels = label_number(scale = 1E-6),
    limits = limits_pretty,
    expand = expansion(mult = c(0, 0.2))
  ) +
  labs(
    x = "Population (millions)"
  )


## ----using-scales, eval = FALSE-----------------------------------------------
# stacked_bar_data |>
#   ggplot(aes(x = continent, y = n_countries, fill = lifeExpGrouped)) +
#   geom_bar(stat = "identity", position = "fill") +
#   theme_af(legend = "right") +
#   scale_y_continuous(
#     expand = expansion(mult = c(0, 0.1)),
#     labels = scales::percent
#   ) +
#   scale_fill_discrete_af() +
#   labs(
#     x = NULL,
#     y = NULL,
#     fill = "Life Expectancy"
#   )

## ----fig.height = 5.5, echo = FALSE-------------------------------------------
stacked_bar_data |>
  ggplot(aes(x = continent, y = n_countries, fill = lifeExpGrouped)) +
  geom_bar(stat = "identity", position = "fill") +
  theme_af(legend = "right") +
  scale_y_continuous(
    expand = expansion(mult = c(0, 0.1)),
    labels = scales::percent
  ) +
  scale_fill_discrete_af() +
  labs(
    x = NULL,
    y = NULL,
    fill = "Life Expectancy"
  )

## ----clip, eval = FALSE-------------------------------------------------------
# last_plot() + coord_cartesian(clip = "off")

## ----fig.height = 5.5, echo = FALSE-------------------------------------------
last_plot() + coord_cartesian(clip = "off")

## ----add-a-line, eval = FALSE-------------------------------------------------
# gapminder |>
#   filter(country == "United Kingdom") |>
#   ggplot(aes(x = year, y = lifeExp)) +
#   geom_line(linewidth = 1, colour = af_colour_values[1]) +
#   geom_hline(
#     yintercept = 75,
#     colour = "#7F7F7F",
#     linewidth = 1,
#     linetype = "dashed"
#   ) +
#   annotate(geom = "text", x = 2007, y = 70, label = "Age 70") +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = expansion(mult = c(0, 0.1))
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 5)) +
#   labs(
#     x = "Year",
#     y = NULL
#   )

## ----echo=FALSE---------------------------------------------------------------
gapminder |>
  filter(country == "United Kingdom") |>
  ggplot(aes(x = year, y = lifeExp)) +
  geom_line(linewidth = 1, colour = af_colour_values[1]) +
  geom_hline(
    yintercept = 75,
    colour = "#7F7F7F",
    linewidth = 1,
    linetype = "dashed"
  ) +
  annotate(geom = "text", x = 2007, y = 70, label = "Age 70") +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 5)) +
  labs(
    x = "Year",
    y = NULL
  )

## ----chart-titles-------------------------------------------------------------
population_chart +
  labs(
    x = NULL,
    y = NULL,
    title = stringr::str_wrap(
      paste("The U.S.A. has the highest population in the Americas"),
      width = 40
    ),
    subtitle = "Population of countries of the Americas (millions), 2007",
    caption = "Source: Gapminder"
  )

## ----chart-titles-yaxis-1-----------------------------------------------------

ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1)),
  ) +
  labs(
    x = NULL,
    y = "Population (millions)",
    title = stringr::str_wrap(
      "The U.S.A. is the most populous country in the Americas",
      35
    ),
    caption = "Source: Gapminder"
  )

## ----chart-titles-y-axi-2-----------------------------------------------------

ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6),
    expand = expansion(mult = c(0, 0.1)),
  ) +
  labs(
    x = NULL,
    y = NULL,
    title = stringr::str_wrap(
      "The U.S.A. is the most populous country in the Americas",
      40
    ),
    subtitle = "Population (millions)",
    caption = "Source: Gapminder"
  )

## ----text-wrap-1--------------------------------------------------------------
plot <-
  ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af() +
  scale_y_continuous(labels = label_number(scale = 1E-6),
                     expand = expansion(mult = c(0, 0.1))) +
  labs(
    x = NULL,
    subtitle = "Population of countries in the Americas, 2007",
    caption = "Source: Gapminder"
  )

plot +
  labs(
    y = "Population in millions",
    title = paste("The U.S.A. is the most populous country in ",
                  "the Americas")
  )

## ----text-wrap-2--------------------------------------------------------------
plot +
  labs(
    y = "Population\nin millions",
    title = stringr::str_wrap(
      paste("The U.S.A. is the most populous country in ",
            "the Americas"),
      width = 40
    )
  )

## ----adjust-theme, eval = FALSE-----------------------------------------------
# ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
#   geom_col(fill = af_colour_values["dark-blue"]) +
#   theme_af(axis = "xy") +
#   theme(
#     axis.line = element_line(colour = "black"),
#     axis.ticks = element_line(colour = "black")
#   ) +
#   scale_y_continuous(
#     expand = expansion(mult = c(0, 0.1)),
#     limits = c(0, 350E6),
#     labels = scales::label_number(scale = 1E-6)
#   ) +
#   labs(
#     x = NULL,
#     y = NULL
#   )

## ----echo = FALSE-------------------------------------------------------------
ggplot(pop_bar_data, aes(x = reorder(country, -pop), y = pop)) +
  geom_col(fill = af_colour_values["dark-blue"]) +
  theme_af(axis = "xy") +
  theme(
    axis.line = element_line(colour = "black"),
    axis.ticks = element_line(colour = "black")
  ) +
  scale_y_continuous(
    expand = expansion(mult = c(0, 0.1)),
    limits = c(0, 350E6),
    labels = scales::label_number(scale = 1E-6)
  ) +
  labs(
    x = NULL,
    y = NULL
  )

## ----html-formatting----------------------------------------------------------
ann_data <- gapminder |>
  filter(country %in% c("United Kingdom", "China")) |>
  mutate(country = factor(country, levels = c("United Kingdom", "China")))

ann_labs <- ann_data |>
  group_by(country) |>
  mutate(min_year = min(year)) |>
  filter(year == max(year)) |>
  ungroup()

ann_data |>
  ggplot(aes(x = year, y = lifeExp, colour = country)) +
  geom_line(linewidth = 1) +
  theme_af(legend = "none") +
  scale_colour_discrete_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = expansion(mult = c(0, 0.1))
  ) +
  scale_x_continuous(
    limits = c(1952, 2017),
    breaks = seq(1952, 2017, 10)
  ) +
  geom_label(
    data = ann_labs,
    aes(x = year, y = lifeExp, label = country),
    size = 4.5,
    hjust = 0,
    vjust = 0.5,
    nudge_x = 0.5,
    linewidth = NA
  ) +
  coord_cartesian(clip = "off") +
  labs(
    x = "Year",
    y = NULL,
    title = "Living Longer",
    subtitle = "Life Expectancy in years in the
    <span style='color:#12436D;'>United Kingdom</span> and
    <span style='color:#F46A25;'>China</span> 1952 to 2007",
    caption = "Source: Gapminder"
  ) +
  theme(plot.subtitle = element_markdown())

## ----af-palette, eval = FALSE-------------------------------------------------
# gapminder |>
#   filter(country %in% c("United Kingdom", "China")) |>
#   ggplot(
#     aes(
#       x = year, y = lifeExp,
#       colour = factor(country, levels = c("United Kingdom", "China"))
#     )
#   ) +
#   geom_line(linewidth = 1) +
#   theme_af(legend = "right") +
#   scale_colour_discrete_af("categorical2") +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = c(0, 0)
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 10)) +
#   labs(
#     x = "Year",
#     y = NULL,
#     colour = NULL
#   )

## ----fig.height = 5, echo = FALSE---------------------------------------------
gapminder |>
  filter(country %in% c("United Kingdom", "China")) |>
  ggplot(
    aes(
      x = year, y = lifeExp,
      colour = factor(country, levels = c("United Kingdom", "China"))
    )
  ) +
  geom_line(linewidth = 1) +
  theme_af(legend = "right") +
  scale_colour_discrete_af("categorical2") +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = c(0, 0)
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 10)) +
  labs(
    x = "Year",
    y = NULL,
    colour = NULL
  )

## ----different-colour-palette-1, eval = FALSE---------------------------------
# my_palette <- c("#0F820D", "#000000")
# 
# gapminder |>
#   filter(country == "United Kingdom") |>
#   ggplot(aes(x = year, y = lifeExp)) +
#   geom_line(linewidth = 1, colour = my_palette[1]) +
#   theme_af() +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = c(0, 0)
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 10)) +
#   labs(
#     x = "Year",
#     y = NULL,
#   )

## ----echo = FALSE-------------------------------------------------------------
my_palette <- c("#0F820D", "#000000")

gapminder |>
  filter(country == "United Kingdom") |>
  ggplot(aes(x = year, y = lifeExp)) +
  geom_line(linewidth = 1, colour = my_palette[1]) +
  theme_af() +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = c(0, 0)
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 10)) +
  labs(
    x = "Year",
    y = NULL,
  )

## ----different-colour-palette-2, eval = FALSE---------------------------------
# gapminder |>
#   filter(country %in% c("United Kingdom", "China")) |>
#   ggplot(
#     aes(
#       x = year, y = lifeExp,
#       colour = factor(country, levels = c("United Kingdom", "China"))
#     )
#   ) +
#   geom_line(linewidth = 1) +
#   theme_af(legend = "right") +
#   scale_colour_manual(values = rev(my_palette)) +
#   scale_y_continuous(
#     limits = c(0, 82),
#     breaks = seq(0, 80, 20),
#     expand = c(0, 0)
#   ) +
#   scale_x_continuous(breaks = seq(1952, 2007, 10)) +
#   labs(
#     x = "Year",
#     y = NULL,
#     colour = NULL
#   )

## ----fig.height = 5.5, echo = FALSE-------------------------------------------
gapminder |>
  filter(country %in% c("United Kingdom", "China")) |>
  ggplot(
    aes(
      x = year, y = lifeExp,
      colour = factor(country, levels = c("United Kingdom", "China"))
    )
  ) +
  geom_line(linewidth = 1) +
  theme_af(legend = "right") +
  scale_colour_manual(values = rev(my_palette)) +
  scale_y_continuous(
    limits = c(0, 82),
    breaks = seq(0, 80, 20),
    expand = c(0, 0)
  ) +
  scale_x_continuous(breaks = seq(1952, 2007, 10)) +
  labs(
    x = "Year",
    y = NULL,
    colour = NULL
  )

