% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/power_sim.R
\name{power_sim}
\alias{power_sim}
\title{Perform a Power Analysis for (Generalized) Linear Mixed-Effects Models via Data Simulation}
\usage{
power_sim(
  formula,
  design,
  test_parameter = NULL,
  fixed_effects,
  random_effects = NULL,
  icc_specs = NULL,
  overall_variance = NULL,
  family = "gaussian",
  power_crit = 0.8,
  n_start,
  n_increment,
  max_simulation_steps = 100,
  n_issue_stop_prop = 0.2,
  n_is_total = TRUE,
  n_sims = 2000,
  alpha = 0.05,
  parallel_plan = "multisession"
)
}
\arguments{
\item{formula}{An lme4-style model formula (e.g. \code{outcome ~ predictor1 * predictor2 + (1 | id)}).}

\item{design}{A \code{PowRPriori_design} object created by \code{define_design()}.}

\item{test_parameter}{A character vector of the variable names to test for power.
If \code{NULL} (default), power is calculated for all fixed effects except the intercept. Note: The parameter names need to
comply with the names expected by the model. Correctly naming of the variables is aided by the output of the \code{get_fixed_effects_structure()} helper function.}

\item{fixed_effects}{A named list of the fixed-effects coefficients. It is highly
recommended to generate this using \code{get_fixed_effects_structure()} or
\code{fixed_effects_from_average_outcome()}.}

\item{random_effects}{A named, nested list specifying the standard deviations (SDs)
and (if applicable) correlations of the random effects. It is highly
recommended to generate this using \code{get_random_effects_structure()}. If this parameter is not used, \code{icc_specs} and \code{overall_variance} need to be supplied.}

\item{icc_specs}{Optional. A named list of Intraclass Correlation Coefficients for
defining simple random-intercept models. Must be used with \code{overall_variance}.}

\item{overall_variance}{The total variance of the outcome, required when \code{icc_specs} is used.}

\item{family}{The model family: \code{"gaussian"} (for LMMs), \code{"binomial"} (for logistic GLMMs),
or \code{"poisson"} (for poisson GLMMs).}

\item{power_crit}{The desired statistical power level (e.g., 0.80 for 80\%).}

\item{n_start}{The starting sample size for the simulation.}

\item{n_increment}{The step size for increasing the sample size in each iteration.}

\item{max_simulation_steps}{A hard stop for the simulation, limiting the number of
sample size steps to prevent infinite loops. Defaults to 100 steps.}

\item{n_issue_stop_prop}{The proportion of model issues (e.g., singular fits,
non-convergence) at which the simulation will be automatically canceled. Defaults to a proportion of 20\%.}

\item{n_is_total}{Boolean that controls how sample sizes are interpreted. If \code{TRUE}
(default), \code{n_start} refers to the total sample size. If \code{FALSE}, it refers to
the sample size per cell (see \code{define_design()} for details on nested designs).}

\item{n_sims}{The number of simulations to run for each sample size step. Defaults to 2000.}

\item{alpha}{The significance level (alpha) for the power calculation. Defaults to 0.05.}

\item{parallel_plan}{A string specifying the \code{future} plan for parallel processing.
Defaults to \code{"multisession"} to enable parallel computing. Use \code{"sequential"} for debugging.}
}
\value{
An object of class \code{PowRPriori}, which is a list containing the power table,
a sample dataset, all simulation parameters, and detailed results from all runs
(coefficients and random effect estimates).
}
\description{
This is the main function of the \code{PowRPriori} package. It iteratively simulates
datasets for increasing sample sizes to determine the required sample size
to achieve a desired level of statistical power for specific model parameters.
}
\details{
The function supports parallel computation using \code{future}.
Simple linear models (i.e. regression models) can also be analyzed using this function. In this case, no specification of the \code{random_effects} or \code{icc_specs}
parameter is necessary.\code{icc_specs} should only be used when simulating a model containing only random intercepts and no random slopes. Refer to the
vignette for a more detailed description of the complete workflow for using this function.
}
\examples{

  design <- define_design(
    id = "subject",
    between = list(group = c("Control", "Treatment")),
    within = list(time = c("pre", "post"))
  )

  fixed_effects <- list(
    `(Intercept)` = 10,
    groupTreatment = 2,
    timepost = 1,
    `groupTreatment:timepost` = 1.5
  )

  random_effects <- list(
    subject = list(`(Intercept)` = 3),
    sd_resid = 5
  )
\donttest{
  power_results <- power_sim(
    formula = y ~ group * time + (1|subject),
    design = design,
    fixed_effects = fixed_effects,
    random_effects = random_effects,
    test_parameter = "groupTreatment:timepost",
    n_start = 20,
    n_increment = 5,
    n_sims = 100, # Use low n_sims for quick examples
    parallel_plan = "multisession"
  )

  summary(power_results)
  plot_sim_model(power_results)
}
}
